
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data configuration
np.random.seed(42)

# Grid
x = np.linspace(-50, 50, 300)
y = np.linspace(-30, 30, 300)
X, Y = np.meshgrid(x, y)

# Player positions (key players on the soccer field)
players = [(5, -5), (-15, 10), (-20, -10), (20, 15), (0, -20)]

# Influence strength function
def influence_strength(x, y, players):
    Z = np.zeros_like(x)
    for px, py in players:
        Z += np.exp(-((x - px) ** 2 + (y - py) ** 2) / 100)
    return Z

# Calculate Z as influence strength
Z = influence_strength(X, Y, players)

# Text variables
title_text = "Soccer Player Influence Field Intensity"
xlabel_text = "Field X Coordinate"
ylabel_text = "Field Y Coordinate"
colorbar_label_text = "Influence Strength"
player_label_format = "Player ({}, {})"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
legend_location = "upper right"
contour_label_format = "%.1f"
fig, ax = plt.subplots(figsize=(12, 7))

# Contour
cnt = ax.contour(X, Y, Z, cmap="Greens", linewidths=1.5)
contour_filled = ax.contourf(X, Y, Z, cmap="Greens", alpha=0.7)
ax.clabel(cnt, cnt.levels, inline=True, fontsize=10, fmt=contour_label_format)

# Player positions
for px, py in players:
    ax.plot(px, py, "ro", markersize=8, label=player_label_format.format(px, py))

# Adding color bar
cbar = fig.colorbar(contour_filled, ax=ax)
cbar.set_label(colorbar_label_text)

# Titles and labels
ax.set_title(title_text, fontsize=16)
ax.set_xlabel(xlabel_text, fontsize=14)
ax.set_ylabel(ylabel_text, fontsize=14)
ax.legend(loc=legend_location, fontsize=10)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("contour_71.pdf", bbox_inches="tight")
