
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)

# Generate data for the plot - representing stress levels over time
x = np.linspace(0, 10, 1000)
conditions = 5
y = [
    np.random.uniform(0.8, 1.2)
    * np.exp(-0.5 * (x - i) ** 2 / np.linspace(0.2, 1, conditions)[index])
    for index, i in enumerate(np.linspace(2, 8, conditions))
]

# Text labels
xlabel = 'Time (hours)'
ylabel = 'Stress Level'
title = 'Stress Level Over Time Under Various Conditions'
cbar_label = 'Condition'

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(12, 6))

# Define a new colormap for the visualization
cmap = plt.cm.viridis
sm = plt.cm.ScalarMappable(cmap=cmap, norm=plt.Normalize(vmin=0, vmax=conditions-1))
cbar = plt.colorbar(sm, ax=ax, label=cbar_label)
cbar.set_label(cbar_label, rotation=270, labelpad=25)

# Plot each condition
for i in range(conditions):
    plt.fill_between(x, y[i], color=cmap(i / conditions), alpha=0.7, label=f'Condition {i+1}')

# Add grid lines for better readability
plt.grid(True, which='both', linestyle='--', linewidth=0.5)

# Add data labels
plt.xlabel(xlabel)
plt.ylabel(ylabel)
plt.title(title)
plt.legend()

# Customize axis appearance
plt.ylim(0, 2)
plt.gca().spines["top"].set_visible(False)
plt.gca().spines["right"].set_visible(False)
plt.gca().spines["bottom"].set_visible(True)
plt.gca().spines["left"].set_visible(False)
plt.gca().set_yticks([])

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_21.pdf", bbox_inches="tight")
