
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Setting seed for reproducibility
np.random.seed(0)

# Months for the x-axis
x = np.linspace(1, 12, 1000)  # Simulating months

# Generate data for 9 different tourist destinations with seasonality
y = [
    np.random.uniform(0.7, 1.3)
    * np.exp(-0.5 * (x - month) ** 2 / np.random.uniform(0.5, 1.5))
    for month in np.linspace(1, 12, 9)
]
cbar_label = "Tourist Destination"

# Text Labels
xlabel = "Month"
ylabel = "Number of Visitors (in thousands)"
title = "Monthly Visitor Trends at Various Tourist Destinations"
legend_labels = [f'Destination {i+1}' for i in range(9)]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(12, 6))

# Create a colorbar
cmap = "viridis"
sm = plt.cm.ScalarMappable(cmap=cmap, norm=plt.Normalize(vmin=0, vmax=9))
cbar = plt.colorbar(sm, ax=ax, label=cbar_label)
cbar.set_label(cbar_label, rotation=270, labelpad=25)

# Plotting the data
for i in range(9):
    plt.fill_between(x, y[i], color=plt.cm.get_cmap(cmap)(i / 9), alpha=0.6, label=legend_labels[i])

# Customizing the plot
plt.ylim(0, 2)
plt.xlabel(xlabel)
plt.ylabel(ylabel)
plt.title(title)
plt.legend(loc="upper right")

# Remove the plot border
plt.gca().spines["top"].set_visible(False)
plt.gca().spines["right"].set_visible(False)
plt.gca().spines["left"].set_visible(False)
plt.gca().set_yticks([])

# Additional aesthetic adjustments
plt.grid(False)
plt.tight_layout()

# ===================
# Part 4: Saving Output
# ===================
# Save the plot
plt.savefig("density_22.pdf", bbox_inches="tight")
