# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation - Media Domain (Streaming Service Usage Data)
# ===================
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(0)

# Generate data for media domain (streaming service usage over time)
y = np.linspace(0, 20, 100)  # Time in months
x1 = np.exp(-0.1 * (y - 10)**2) + 0.1 * np.random.rand(100)  # Users for Streaming Service A
x2 = 0.75 * np.exp(-0.15 * (y - 12)**2) + 0.1 * np.random.rand(100)  # Users for Streaming Service B

# Combine data for density estimation
data_service_A = x1
data_service_B = x2

# Kernel Density Estimation (KDE) for the streaming service usage data
kde_service_A = gaussian_kde(data_service_A, bw_method='scott')
kde_service_B = gaussian_kde(data_service_B, bw_method='scott')

x_dens_service = np.linspace(0, max(data_service_A.max(), data_service_B.max()), 1000)
density_service_A = kde_service_A(x_dens_service)
density_service_B = kde_service_B(x_dens_service)

# Extracted variables for the density plot
xlabel_text = "Number of Users (in Thousands)"
ylabel_text = "Density"
title_text = "Density of Streaming Service Usage Over Time"
label=["Streaming Service A Density","Streaming Service B Density"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

# Plot the density curves for streaming service usage data
ax.plot(x_dens_service, density_service_A, color='#3498db', lw=2, label=label[0])
ax.fill_between(x_dens_service, density_service_A, color='#3498db', alpha=0.3)

ax.plot(x_dens_service, density_service_B, color='#e67e22', lw=2, label=label[1])
ax.fill_between(x_dens_service, density_service_B, color='#e67e22', alpha=0.3)

# Set labels and title
ax.set_xlabel(xlabel_text, fontsize=12)
ax.set_ylabel(ylabel_text, fontsize=12)
ax.set_title(title_text, fontsize=16)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.grid(True, linestyle=":", alpha=0.5)

# Adding a legend
ax.legend(loc='upper right')

# ===================
# Part 4: Saving Output for Density Plot
# ===================
plt.tight_layout()
plt.savefig("density_26.pdf", bbox_inches="tight")
