# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation - AQI Data
# ===================
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(24)
x = np.linspace(0, 14, 800)  # Time in weeks
# Simulated AQI data for seven different cities
aqi_data = [
    np.abs(np.random.uniform(50, 150) * (0.5 + 0.5 * np.sin(0.3 * (x - i))))
    for index, i in enumerate(np.linspace(0, 6, 7))
]

# Combine data for density estimation
combined_aqi = np.concatenate(aqi_data)

# Kernel Density Estimation (KDE) for the AQI data
kde_aqi = gaussian_kde(combined_aqi, bw_method='scott')
x_dens_aqi = np.linspace(0, max(combined_aqi), 1000)
density_aqi = kde_aqi(x_dens_aqi)

# Create density plots for individual city data
density_curves = []
for aqi in aqi_data:
    kde_city = gaussian_kde(aqi, bw_method='scott')
    density_curves.append(kde_city(x_dens_aqi))

# Extracted variables for the density plot
xlabel_text = "Air Quality Index (AQI)"
ylabel_text = "Density"
title_text = "Density of Air Quality Index (AQI) Across Different Cities"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 5))

# Plot the density curve for overall AQI data
ax.plot(x_dens_aqi, density_aqi, color='purple', lw=2, label='Overall AQI Density')
ax.fill_between(x_dens_aqi, density_aqi, color='purple', alpha=0.3)

# Plot density area for an additional city
additional_city_data = np.abs(np.random.uniform(40, 130) * (0.5 + 0.5 * np.sin(0.4 * (x - 4))))
kde_additional_city = gaussian_kde(additional_city_data, bw_method='scott')
density_additional_city = kde_additional_city(x_dens_aqi)
ax.fill_between(x_dens_aqi, density_additional_city, color='orange', alpha=0.5, label='City 8 AQI Density')

# Set labels and title
ax.set_xlabel(xlabel_text, fontsize=12)
ax.set_ylabel(ylabel_text, fontsize=12)
ax.set_title(title_text, fontsize=16)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.grid(True, linestyle=":", alpha=0.5)

# Adding a legend
ax.legend(loc='upper right', fontsize=10)

# ===================
# Part 4: Saving Output for Density Plot
# ===================
plt.tight_layout()
plt.savefig("density_36.pdf", bbox_inches="tight")
