# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation - City Temperature Data
# ===================
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(42)
cities = ['New York', 'Los Angeles', 'Chicago', 'Houston', 'Phoenix']
# Simulated temperature data for each city over a year (in Celsius)
temperatures = [
    np.random.normal(15 + i*3, 5, 365)  # Mean temperature varies by city
    for i in range(len(cities))
]

# Adding another set of temperature data for variation
more_temperatures = [
    np.random.normal(20 + i*2, 4, 365)  # Another set with different mean and variance
    for i in range(3)  # Only add a few extra cities to avoid too many lines
]

# Combine data for density estimation
temp_data = np.concatenate(temperatures + more_temperatures)

# Kernel Density Estimation (KDE) for the temperature data
kde_temp = gaussian_kde(temp_data, bw_method='scott')
x_dens_temp = np.linspace(min(temp_data), max(temp_data), 1000)
density_temp = kde_temp(x_dens_temp)

# Create density plots for individual city data
density_curves = []
for temp in temperatures + more_temperatures:
    kde_city = gaussian_kde(temp, bw_method='scott')
    density_curves.append(kde_city(x_dens_temp))

# Extracted variables for the density plot
xlabel_text = "Temperature (°C)"
ylabel_text = "Density"
title_text = "Density of Yearly Temperatures Across Different Cities"
label ="Overall Temperature Density"


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))

# Plot the density curve for overall temperature data
ax.plot(x_dens_temp, density_temp, color='blue', lw=2, label=label)
ax.fill_between(x_dens_temp, density_temp, color='blue', alpha=0.3)

# Plot individual city density curves (fewer lines)
colors = ['#1f77b4', '#ff7f0e', '#2ca02c', '#d62728', '#9467bd']
for i, density in enumerate(density_curves):
    ax.plot(x_dens_temp, density, color=colors[i % len(colors)], lw=1.5, alpha=0.7, label=f'City {i+1} Density')

# Set labels and title
ax.set_xlabel(xlabel_text, fontsize=12)
ax.set_ylabel(ylabel_text, fontsize=12)
ax.set_title(title_text, fontsize=16)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.grid(True, linestyle=":", alpha=0.5)

# Adding a legend
ax.legend(loc='upper right', fontsize=10)

# ===================
# Part 4: Saving Output for Density Plot
# ===================
plt.tight_layout()
plt.savefig("density_43.pdf", bbox_inches="tight")
