
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Create transportation data
years = np.linspace(2000, 2035, 100)  # Years from 2000 to 2035
electric_car_adoption = 0.8 * np.exp(-0.002 * (years - 2030) ** 2)  # Peaks around 2030
public_transport_ridership = 0.7 * (1 / (1 + np.exp(-0.1 * (years - 2020))))  # S-curve growth

# Text variables
xlabel_text = "Year"
ylabel_text = "Proportion of Total Usage"
title_text = "Trends in Transportation: Electric Cars vs. Public Transport"
legend_electric_car = "Electric Car Adoption"
legend_public_transport = "Public Transport Ridership"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
legend_location = "upper left"
# Create the figure and axis
fig, ax = plt.subplots(figsize=(10, 6))

# Plot the data
ax.fill_between(
    years,
    electric_car_adoption,
    color="#66c2a5",
    edgecolor="#1b7837",
    alpha=0.6,
    label=legend_electric_car,
)
ax.fill_between(
    years,
    public_transport_ridership,
    color="#3288bd",
    edgecolor="#08519c",
    alpha=0.6,
    label=legend_public_transport,
)

# Customize the plot
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["left"].set_visible(True)
ax.spines["bottom"].set_visible(True)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.set_xlabel(xlabel_text)
ax.set_ylabel(ylabel_text)
ax.set_title(title_text)
ax.set_ylim(0, 1)
ax.grid(True, which='both', linestyle='--', linewidth=0.5, color='gray', alpha=0.7)

# Add legend
ax.legend(loc=legend_location, frameon=False)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_59.pdf", bbox_inches="tight")
