
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
from scipy.stats import gaussian_kde

# Set random seed for reproducibility
np.random.seed(101)

# Generate a societal domain dataset simulating income distribution
low_income = np.random.normal(loc=20000, scale=5000, size=500)
mid_income = np.random.normal(loc=45000, scale=10000, size=1000)
high_income = np.random.normal(loc=100000, scale=20000, size=150)
income_data = np.concatenate([low_income, mid_income, high_income])
xs = np.linspace(0, 150000, 300)

# Axes Limits and Labels
title = "KDE Plot of Simulated Income Distribution"
xlabel_value = "Income Amount (in dollars)"
ylabel_value = "Density"

xticks_values = [0, 30000, 60000, 90000, 120000, 150000]
xticklabels = ["0", "30K", "60K", "90K", "120K", "150K"]
yticks_values = [0, 2e-05, 4e-05, 6e-05, 8e-05]
yticklabels = ["0", "2e-05", "4e-05", "6e-05", "8e-05"]
xlim_values = [0, 150000]
ylim_values = [0, 9e-05]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

# Set the figure size
fig, ax = plt.subplots(figsize=(12, 6))

# Create the KDE plot with adjusted x-axis range
density = gaussian_kde(income_data)
density.covariance_factor = lambda: 0.25
density._compute_covariance()
plt.fill_between(xs, density(xs), color="#add8e6", edgecolor="#4682b4")

ax.set_xticks(xticks_values)
ax.set_xticklabels(xticklabels)

ax.set_yticks(yticks_values)
ax.set_yticklabels(yticklabels)

plt.xlim(xlim_values)
plt.ylim(ylim_values)
# Set the title and labels
plt.title(title, fontsize=14)
plt.xlabel(xlabel_value, fontsize=12)
plt.ylabel(ylabel_value, fontsize=12)

# Add grid for better readability
ax.grid(True, which='both', linestyle='--', linewidth=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_64.pdf", bbox_inches="tight")
