
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generate sample population age distributions for three different cities
city_a_ages = np.random.normal(loc=30, scale=8, size=1000)
city_b_ages = np.random.normal(loc=40, scale=10, size=1000)
city_c_ages = np.random.normal(loc=50, scale=12, size=1000)

labels = ["City A Age Distribution", "City B Age Distribution", "City C Age Distribution"]
avxlabel = "Average Age in City A"
xlabel = "Age"
ylabel = "Density"
title = "Age Distribution Density for Three Cities"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
grid_linestyle = "-."
grid_linewidth = 0.7
grid_alpha = 0.8
line_linestyle = "-"
line_linewidth = 2.5
fill_alpha = 0.3

fig, ax = plt.subplots(figsize=(10, 6))

# Define new colors
colors = ["#3498db", "#e74c3c", "#2ecc71"]

# Plot the density plots
for data, color, label in zip(
    [city_a_ages, city_b_ages, city_c_ages],
    colors,
    labels,
):
    density = gaussian_kde(data)
    xs = np.linspace(min(data) - 10, max(data) + 10, 200)
    density.covariance_factor = lambda: 0.35  # Adjust for smoother curves
    density._compute_covariance()
    plt.plot(
        xs,
        density(xs),
        color=color,
        label=label,
        linestyle=line_linestyle,  # solid line
        linewidth=line_linewidth,  # slightly thicker lines
    )
    plt.fill_between(
        xs, density(xs), color=color, alpha=fill_alpha
    )  # strong fill for shade

# Plot a vertical line indicating the average age in City A
plt.axvline(
    x=np.mean(city_a_ages),
    color="#9b59b6",
    linestyle=line_linestyle,
    linewidth=line_linewidth,
    label=avxlabel,
)

# Set labels and title
ax.set_xlim(
    min(min(city_a_ages), min(city_b_ages), min(city_c_ages)) - 10,
    max(max(city_a_ages), max(city_b_ages), max(city_c_ages)) + 10,
)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Show grid with improved linestyle
plt.grid(True, linestyle=grid_linestyle, linewidth=grid_linewidth, alpha=grid_alpha)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("density_66.pdf", bbox_inches="tight")

