
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(24)

# Generate sample music streaming data
pop_streams = np.random.normal(loc=1500, scale=300, size=1000)  # in streams
rock_streams = np.random.normal(loc=1200, scale=250, size=1000)  # in streams
jazz_streams = np.random.normal(loc=800, scale=200, size=1000)  # in streams

labels = ["Pop Streams", "Rock Streams", "Jazz Streams"]
avxlabel = "Average Pop Streams"
xlabel = "Number of Streams"
ylabel = "Density"
title = "Music Genre Streaming Density Distributions"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
grid_linestyle = "-"
grid_linewidth = 0.5
grid_alpha = 0.7
line_linestyle = "-."
line_linewidth = 2
fill_alpha = 0.25

fig, ax = plt.subplots(figsize=(10, 6))

# Plot the density plots
for data, color, label in zip(
    [pop_streams, rock_streams, jazz_streams],
    ["magenta", "cyan", "limegreen"],
    labels,
):
    density = gaussian_kde(data)
    xs = np.linspace(min(data) - 50, max(data) + 50, 200)
    density.covariance_factor = lambda: 0.25  # Adjust for smoother curves
    density._compute_covariance()
    plt.plot(
        xs,
        density(xs),
        color=color,
        label=label,
        linestyle=line_linestyle,  # dash-dot line
        linewidth=line_linewidth,  # slightly thicker lines
    )
    plt.fill_between(
        xs, density(xs), color=color, alpha=fill_alpha
    )  # fill for shade

# Plot a vertical line indicating the average pop streams
plt.axvline(
    x=np.mean(pop_streams),
    color="red",
    linestyle=line_linestyle,
    linewidth=line_linewidth,
    label=avxlabel,
)

# Set labels and title
ax.set_xlim(
    min(min(pop_streams), min(rock_streams), min(jazz_streams)) - 50,
    max(max(pop_streams), max(rock_streams), max(jazz_streams)) + 50,
)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Show grid with a thicker linestyle
plt.grid(True, linestyle=grid_linestyle, linewidth=grid_linewidth, alpha=grid_alpha)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("density_67.pdf", bbox_inches="tight")
