
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.colors as mcolors

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Data for different historical battles
categories = [
    "Battle of Waterloo",
    "Battle of Gettysburg",
    "Battle of Stalingrad",
    "Battle of Hastings",
    "Battle of Thermopylae",
    "Battle of Agincourt",
    "Battle of Tours",
]

survivor_counts = [
    15000,
    23000,
    50000,
    8000,
    10000,
    12000,
    18000,
]  # Survivors in each battle
casualty_counts = [
    -25000,
    -28000,
    -100000,
    -20000,
    -12000,
    -8000,
    -20000,
]  # Casualties in each battle (negative values)

# Error data for each category
survivor_errors = [1000, 1500, 5000, 800, 500, 600, 1200]
casualty_errors = [2000, 3000, 10000, 2000, 1500, 1000, 2500]

# Labels for the chart
xlabel = "Number of People"
ylabel = "Historical Battles"
title = "Casualties and Survivors in Historical Battles"
legend_labels = ["Casualties", "Survivors"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Define colors
survivor_colors = [
    "#8c8c8c",
    "#b0b0b0",
    "#a0a0a0",
    "#c0c0c0",
    "#d0d0d0",
    "#e0e0e0",
    "#f0f0f0",
]

casualty_colors = [
    mcolors.to_rgba(color, alpha=0.6) for color in survivor_colors
]

# Create figure and axis
fig, ax = plt.subplots(figsize=(10, 7))  # Adjusted size for clarity

# Plot horizontal bar charts with error bars
ax.barh(
    categories,
    casualty_counts,
    color=casualty_colors,
    edgecolor="black",
    xerr=casualty_errors,
    label=legend_labels[0],
    capsize=3,
)
ax.barh(
    categories,
    survivor_counts,
    left=0,
    color=survivor_colors,
    edgecolor="black",
    xerr=survivor_errors,
    label=legend_labels[1],
    capsize=3,
)

# Adding labels within bars for values
for i, (cas_val, surv_val) in enumerate(zip(casualty_counts, survivor_counts)):
    ax.text(cas_val / 2, i, f"{abs(cas_val)}", va="center", ha="center", color="black")
    ax.text(surv_val / 2, i, f"{surv_val}", va="center", ha="center", color="black")

# Customizing axis and layout
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)
ax.tick_params(axis="y", which="both", left=False)  # Remove y-axis tick marks
ax.set_xlim(min(casualty_counts) - 10000, max(survivor_counts) + 10000)
ax.spines["left"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["top"].set_visible(False)
ax.axvline(0, color="black")  # Draw a vertical line at x=0 for separation

# Labels at the ends
ax.text(min(casualty_counts) - 15000, len(categories), legend_labels[0], ha="right", va="bottom")
ax.text(max(survivor_counts) + 5000, len(categories), legend_labels[1], ha="left", va="bottom")

# Invert the y-axis for readability
ax.invert_yaxis()
ax.legend()

# ===================
# Part 4: Saving Output
# ===================
# Show plot with tight layout
plt.tight_layout()
plt.savefig("errorbar_106.pdf", bbox_inches="tight")
