
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Data
critics = ["Critic A", "Critic B", "Critic C", "Critic D"]
art_styles = [
    "Impressionism",
    "Cubism",
    "Surrealism",
    "Abstract",
    "Pop Art",
    "Modernism",
]
popularity_means = np.array(
    [
        [75, 60, 65, 70, 80, 90],  # Critic A
        [70, 55, 60, 65, 75, 85],  # Critic B
        [85, 70, 75, 80, 85, 95],  # Critic C
        [80, 65, 70, 75, 80, 90],  # Critic D
    ]
)
popularity_std = np.array(
    [
        [5, 4, 3, 5, 4, 6],  # Critic A
        [4, 3, 4, 4, 3, 5],  # Critic B
        [5, 4, 5, 3, 4, 6],  # Critic C
        [4, 3, 4, 5, 3, 5],  # Critic D
    ]
)
legendtitle = "Art Style"
xlabel = "Critic"
ylabel = "Popularity (%)"
ylim = [0, 100]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set figure size to match the original image's dimensions
fig, ax = plt.subplots(figsize=(12, 7))

# New artistic colors inspired by famous art movements
colors = [
    "#e69f00",  # Impressionism (Orange)
    "#56b4e9",  # Cubism (Sky Blue)
    "#009e73",  # Surrealism (Green)
    "#f0e442",  # Abstract (Yellow)
    "#0072b2",  # Pop Art (Blue)
    "#d55e00",  # Modernism (Red)
]

# Bar width
bar_width = 0.15
bar_width_primary = 0.75

# Set position of bar on X axis
r = np.arange(len(critics))

# Draw bars for 'Impressionism' art style
i = art_styles.index("Impressionism")
ax.bar(
    r + (i + 3) * bar_width,
    popularity_means[:, i],
    yerr=popularity_std[:, i],
    width=bar_width_primary,
    label=art_styles[i],
    capsize=5,
    color=colors[i],
    hatch="///",
    edgecolor="black",
)

# Draw bars for other art styles
for i in range(len(art_styles)):
    if art_styles[i] == "Impressionism":
        continue
    ax.bar(
        r + i * bar_width,
        popularity_means[:, i],
        yerr=popularity_std[:, i],
        width=bar_width,
        label=art_styles[i],
        capsize=5,
        color=colors[i],
        edgecolor="black",
    )

# Add xticks on the middle of the group bars
ax.set_xlabel(xlabel)
ax.set_xticks(r + bar_width * (len(art_styles) - 1) / 2)
ax.set_xticklabels(critics)

# Create legend & Show graphic
handles, labels = ax.get_legend_handles_labels()
order = [0, 1, 2, 4, 3, 5]  # Reordering the legend
ax.legend(
    [handles[idx] for idx in order],
    [labels[idx] for idx in order],
    loc="upper center",
    bbox_to_anchor=(0.5, 1.15),
    ncol=6,
    title=legendtitle,
)
ax.set_ylabel(ylabel)
ax.set_ylim(ylim)  # Adjust y-axis limit to accommodate error bars

# Improve aesthetics
ax.grid(True, which='both', linestyle='--', linewidth=0.5)
ax.set_facecolor('#f7f7f7')

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("errorbar_35.pdf", bbox_inches="tight")
