
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Seed for reproducibility
np.random.seed(42)

# Categories of expenditures and their labels
expenditure_types = ["Housing", "Food", "Healthcare"]

# Monthly x positions
x_positions = np.linspace(-20, 20, 6)
housing_expenditure = np.random.uniform(1000, 1500, 6)
food_expenditure = np.random.uniform(300, 600, 6)
healthcare_expenditure = np.random.uniform(200, 400, 6)

# Random error margins (positive and negative)
housing_error = [np.random.uniform(50, 100, 6), np.random.uniform(50, 100, 6)]
food_error = [np.random.uniform(20, 30, 6), np.random.uniform(20, 30, 6)]
healthcare_error = np.random.uniform(10, 20, 6)

# Label positions
vertical_line_position = 2.5

# Titles for each subplot
titles = ["Average Housing Expenditure", "Average Food Expenditure", "Average Healthcare Expenditure"]

# Axis labels
xlabel = "Months"
ylabel = "Expenditure in USD ($)"

suptitle = "Average Monthly Expenditure Over Six Months"
filename = "errorpoint_19_4.pdf"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with three subplots and a shared x-axis
fig, (ax0, ax1, ax2) = plt.subplots(figsize=(8, 10), nrows=3, sharex=True)

# Color maps and marker styles
housing_colors = plt.get_cmap("Pastel1")(np.linspace(0, 1, 6))
food_colors = plt.get_cmap("Pastel2")(np.linspace(0, 1, 6))
healthcare_color = "#8dd3c7"

# First subplot with symmetric vertical error bars
for i in range(len(x_positions)):
    ax0.errorbar(
        x_positions[i],
        housing_expenditure[i],
        yerr=[[housing_error[0][i]], [housing_error[1][i]]],
        fmt="o",
        color=housing_colors[i],
        capsize=4,
    )
    ax0.text(
        x_positions[i] - 0.1,
        housing_expenditure[i] + 20,
        f"{housing_expenditure[i]:.2f}",
        fontsize=8,
        ha="right",
    )
ax0.set_title(titles[0])
ax0.axhline(y=1300, linestyle="--", color="#FF6347")
ax0.yaxis.grid(True)
ax0.xaxis.grid(False)

# Second subplot with symmetric horizontal error bars
for i in range(len(x_positions)):
    ax1.errorbar(
        x_positions[i],
        food_expenditure[i],
        xerr=[[food_error[0][i]], [food_error[1][i]]],
        fmt="s",
        color=food_colors[i],
        capsize=4,
    )
    ax1.text(
        x_positions[i] + 0.1,
        food_expenditure[i] + 5,
        f"{food_expenditure[i]:.2f}",
        fontsize=8,
        ha="left",
    )
ax1.set_title(titles[1])
ax1.axvline(x=vertical_line_position, linestyle="--", color="#4682B4")
ax1.xaxis.grid(True)
ax1.yaxis.grid(False)

# Third subplot with symmetric vertical error bars
ax2.errorbar(
    x_positions,
    healthcare_expenditure,
    yerr=healthcare_error,
    fmt="*",
    color=healthcare_color,
    capsize=4,
)
ax2.set_title(titles[2])
ax2.legend([expenditure_types[2]], loc="upper left")
ax2.yaxis.grid(True)
ax2.xaxis.grid(False)

# Set common labels
fig.supxlabel(xlabel)
fig.supylabel(ylabel)
fig.suptitle(suptitle, fontsize=14)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout(rect=[0, 0, 1, 0.95])
plt.savefig("errorpoint_76.pdf", bbox_inches="tight")
