
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np  # Moved from Part 2
np.random.seed(42)  # Ensuring reproducibility
# Create a random graph to represent a transportation network
G = nx.random_geometric_graph(50, 0.25)

# Position nodes using the Kamada-Kawai layout for better visualization
pos = nx.kamada_kawai_layout(G)

# Node type assignment: 50% major hubs (blue), rest transit stops (green)
node_colors = []
for node in G.nodes():
    if np.random.rand() > 0.5:
        node_colors.append("blue")  # Major hub
    else:
        node_colors.append("green")  # Transit stop

# Edges: Mark 30% of the edges as undergoing maintenance (dashed lines, highlighted)
edges = list(G.edges())
num_maintenance = int(len(edges) * 0.3)
maintenance_edges = np.random.choice(len(edges), size=num_maintenance, replace=False)
maintenance_edges = [edges[i] for i in maintenance_edges]

title = "Transportation Network"
labels = ["Transit Stop", "Major Hub", "Operational Route", "Route Under Maintenance"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Draw the nodes with respective colors
nx.draw_networkx_nodes(G, pos, node_size=100, node_color=node_colors, ax=ax)

# Draw all edges
nx.draw_networkx_edges(G, pos, ax=ax, edge_color="gray", alpha=0.5)

# Draw edges under maintenance with a different style (dashed)
nx.draw_networkx_edges(
    G, pos, ax=ax, edgelist=maintenance_edges, style="dashed", edge_color="red"
)

# Add a legend
legend_elements = [
    plt.Line2D(
        [0],
        [0],
        marker="o",
        color="w",
        markerfacecolor="green",
        markersize=10,
        label=labels[0],
    ),
    plt.Line2D(
        [0],
        [0],
        marker="o",
        color="w",
        markerfacecolor="blue",
        markersize=10,
        label=labels[1],
    ),
    plt.Line2D([0], [0], color="gray", lw=2, alpha=0.5, label=labels[2]),
    plt.Line2D([0], [0], color="red", linestyle="--", linewidth=2, label=labels[3]),
]

ax.legend(handles=legend_elements, loc="upper right")

# Remove axis
plt.axis("off")

# Add a title
plt.title(title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("graph_46.pdf", bbox_inches="tight")
