
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)  # Ensuring reproducibility
# Create a random graph to represent a healthcare network
G = nx.random_geometric_graph(50, 0.2)

# Position nodes using the Kamada-Kawai layout to achieve a more spread-out layout
pos = nx.kamada_kawai_layout(G)

# Node type assignment: Hospitals (green), Clinics (yellow), Mobile Health Units (purple)
node_colors = []
for node in G.nodes():
    rand_val = np.random.rand()
    if rand_val < 0.33:
        node_colors.append("green")  # Hospital
    elif rand_val < 0.66:
        node_colors.append("yellow")  # Clinic
    else:
        node_colors.append("purple")  # Mobile Health Unit

# Edges: Mark 40% of the edges as experiencing delays (dotted lines, highlighted)
edges = list(G.edges())
num_delayed_edges = int(len(edges) * 0.4)
delayed_edges_indices = np.random.choice(len(edges), size=num_delayed_edges, replace=False)
delayed_edges = [edges[i] for i in delayed_edges_indices]

title = "Healthcare Network"
labels = ["Hospital", "Clinic", "Mobile Health Unit", "Regular Connection", "Delayed Connection"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Draw the nodes with respective colors
nx.draw_networkx_nodes(G, pos, node_size=150, node_color=node_colors, ax=ax)

# Draw all edges
nx.draw_networkx_edges(G, pos, ax=ax, edge_color="gray", alpha=0.5, width=1.5)

# Draw delayed edges with a different style (dotted)
nx.draw_networkx_edges(
    G, pos, ax=ax, edgelist=delayed_edges, style="dotted", edge_color="red", width=2
)

# Add a legend
legend_elements = [
    plt.Line2D([0], [0], marker="o", color="w", markerfacecolor="green", markersize=10, label=labels[0]),
    plt.Line2D([0], [0], marker="o", color="w", markerfacecolor="yellow", markersize=10, label=labels[1]),
    plt.Line2D([0], [0], marker="o", color="w", markerfacecolor="purple", markersize=10, label=labels[2]),
    plt.Line2D([0], [0], color="gray", lw=2, alpha=0.5, label=labels[3]),
    plt.Line2D([0], [0], color="red", linestyle=":", linewidth=2, label=labels[4]),
]

ax.legend(handles=legend_elements, loc="upper right")

# Remove axis
plt.axis("off")

# Add a title
plt.title(title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("graph_47.pdf", bbox_inches="tight")
