
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
import networkx as nx

np.random.seed(42)
# Create a random graph representing a communication network
G = nx.random_geometric_graph(30, 0.4)

# Position the nodes using the Kamada-Kawai layout algorithm for a spread-out structure
pos = nx.kamada_kawai_layout(G)

# Randomly select some edges to color as 'high-frequency communication links'
edges = list(G.edges())
high_freq_comms = np.random.choice(
    len(edges), size=int(len(edges) * 0.25), replace=False
)
high_freq_comms = [edges[i] for i in high_freq_comms]

title = "Communication Network"
labels = ["High-frequency Communication", "Communication Links"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 12))

# Draw the nodes with a gradient color
node_colors = np.linspace(0.4, 1.0, len(G.nodes()))
nx.draw_networkx_nodes(
    G, pos, node_size=300, node_color=node_colors, cmap=plt.cm.Blues, edgecolors="#1f3a93", linewidths=0.7
)

# Draw the edges with a light transparency
nx.draw_networkx_edges(G, pos, edge_color="#87CEEB", alpha=0.6)

# Draw the selected 'high-frequency communication' edges with a distinct color
nx.draw_networkx_edges(
    G, pos, edgelist=high_freq_comms, edge_color="#800080", width=2.5
)

# Add a title
plt.title(title, fontsize=18)

# Add a legend
high_freq_patch = plt.Line2D(
    [], [], color="#800080", linewidth=2.5, linestyle="-", label=labels[0]
)
comm_patch = plt.Line2D(
    [], [], color="#87CEEB", linewidth=2, linestyle="-", alpha=0.6, label=labels[1]
)
plt.legend(handles=[high_freq_patch, comm_patch], fontsize=12)

# Remove axis
plt.axis("off")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("graph_48.pdf", bbox_inches="tight")
