
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import networkx as nx
import numpy as np

# Create a directed graph representing an economic relationship network
G = nx.DiGraph()

# Add nodes representing different entities in the economy
entities = [1, 0, 2, 4, 3]
G.add_nodes_from(entities)

# Add edges representing economic interactions (trade agreements)
trade_agreements = [(0, 1), (0, 2), (1, 3), (2, 4), (3, 4)]

# Add self-loops representing internal economic activities
internal_activities = [(0, 0), (1, 1), (2, 2), (3, 3), (4, 4)]

# Combine all edges
G.add_edges_from(trade_agreements + internal_activities)

# Position nodes using a circular layout
pos = nx.circular_layout(G)

# Titles and Annotations
title = "Economic Relationship Network"
annotation_text = "Solid Lines: Trade Agreements\nDashed Lines: Internal Activities"
xlabel = "X-axis: Network Layout"
ylabel = "Y-axis: Network Layout"
legend_labels = ["Trade Agreements", "Internal Activities"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(10, 8))

# Draw the graph with custom node and edge colors
nx.draw(G, pos, with_labels=True, node_color="seagreen", edge_color="gray", node_size=700)

# Highlight the trade agreement edges with solid lines
nx.draw_networkx_edges(
    G,
    pos,
    edgelist=trade_agreements,
    edge_color="royalblue",
    style="solid",
    width=2.5,
    arrowstyle="-|>",
)

# Highlight the internal activities with dashed lines
nx.draw_networkx_edges(
    G, pos, edgelist=internal_activities, edge_color="goldenrod", style="dashed", width=2, arrowstyle="-|>"
)

# Add a title and annotations to the plot
plt.title(title, fontsize=16, fontweight='bold')
props = dict(boxstyle="round", facecolor="lightyellow", alpha=0.6)
plt.text(
    0.05,
    0.95,
    annotation_text,
    transform=plt.gca().transAxes,
    fontsize=12,
    verticalalignment="top",
    bbox=props,
)

# Axis labels
plt.xlabel(xlabel, fontsize=14)
plt.ylabel(ylabel, fontsize=14)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_54.pdf", bbox_inches="tight")
