
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Create a directed graph
G = nx.DiGraph()

# Add nodes representing different entities in the social flow
nodes = [
    "City Hall",
    "Local Library",
    "Community Center",
    "Neighborhood Association",
    "Residents",
]
for node in nodes:
    G.add_node(node)

# Define edges representing information flow between entities
edges = [
    ("City Hall", "Local Library"),
    ("City Hall", "Community Center"),
    ("Community Center", "Neighborhood Association"),
    ("Neighborhood Association", "Residents"),
    (
        "Local Library",
        "Local Library",
    ),  # Self-loop for data archiving or events
    (
        "Residents",
        "Residents",
    ),  # Self-loop for social discussions or self-organization
    (
        "City Hall",
        "City Hall",
    ),  # Self-loop for governmental processes or feedback
]

# Add edges to the graph
G.add_edges_from(edges)

# Define positions for a clear layout
pos = nx.spring_layout(G, seed=42)

title = "Information Flow in a Community Network"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(12, 10))

# Define a new color scheme for the nodes
node_colors = [
    "green" if node == "City Hall" else "blue" if node == "Local Library" 
    else "yellow" if node == "Community Center" else "orange" if node == "Neighborhood Association" 
    else "purple" for node in G.nodes()
]

# Draw the graph with customized node colors and labels
nx.draw(
    G,
    pos,
    with_labels=True,
    node_color=node_colors,
    node_size=3000,
    font_size=10,
    font_weight="bold",
    edge_color="gray",
)

# Draw edges with differentiated styles for self-loops
self_loops = [(u, v) for u, v in G.edges() if u == v]
other_edges = [(u, v) for u, v in G.edges() if u != v]

nx.draw_networkx_edges(
    G,
    pos,
    edgelist=other_edges,
    arrowstyle="-|>",
    arrowsize=20,
    width=2.0,
    edge_color="black",
)
nx.draw_networkx_edges(
    G,
    pos,
    edgelist=self_loops,
    arrowstyle="-|>",
    style="dashed",
    arrowsize=20,
    width=2.0,
    edge_color="red",
)

# Title and labels
plt.title(title, fontsize=15)
plt.axis("off")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_62.pdf", bbox_inches="tight")

