
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx
import random

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Create a random graph to simulate population migration between 15 cities
random.seed(20)
np.random.seed(42)  # For reproducibility
G = nx.erdos_renyi_graph(15, 0.2)
weights = {edge: np.random.randint(100, 1000) for edge in G.edges()}  # Migration numbers between cities
nx.set_edge_attributes(G, weights, "weight")

# Use a spring layout for node positions
pos = nx.spring_layout(G, seed=42)

# Labels for nodes (cities)
labels = {i: f"City {i}" for i in G.nodes()}

# Draw edge labels (migration numbers)
edge_labels = nx.get_edge_attributes(G, "weight")

# Simulating population densities for node sizes
population = {i: np.random.randint(50000, 1000000) for i in G.nodes()}
nx.set_node_attributes(G, population, "population")

# Title and other texts
title = "Population Migration Network"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(14, 12))

# Custom color scheme for nodes and edges
node_colors = "forestgreen"
edge_colors = "royalblue"

# Draw nodes with sizes based on simulated population
nx.draw(
    G, pos, node_size=[population[i] * 0.00005 for i in G.nodes()], node_color=node_colors, 
    edge_color=edge_colors, with_labels=False
)

# Draw the labels for the nodes
nx.draw_networkx_labels(
    G, pos, labels=labels, font_color="black", font_size=12, font_weight="bold"
)

# Draw the edges with varying widths based on migration weights
nx.draw_networkx_edges(
    G, pos, width=[weights[edge] * 0.005 for edge in G.edges()], edge_color=edge_colors
)

# Draw edge labels with weights
nx.draw_networkx_edge_labels(
    G, pos, edge_labels=edge_labels, font_color="red", font_size=10
)

# Add a title and configure layout
plt.title(title, size=22)
plt.tight_layout()

# ===================
# Part 4: Saving Output
# ===================
# Save the plot
plt.savefig("graph_78.pdf", bbox_inches="tight")
