
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib
import matplotlib as mpl

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
cities = ["New York", "Los Angeles", "Chicago", "Houston", "Phoenix", "Philadelphia", "San Antonio"]
transportation = ["Cars", "Buses", "Trains", "Planes"]

vehicles_count = np.random.randint(50, 500, size=(len(cities), len(transportation)))

cbarlabels = ["Vehicles Count", "Yearly Transport Count", "Efficiency Rate"]

# Label parameters
xlabel = "Cities"
ylabel = "Transportation Types"
chart_title = "Transportation Vehicle Distribution by City"
suptitle = "Heatmaps of Various Transportation Metrics"
legendlabels = ["Cars", "Buses", "Trains", "Planes"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

def heatmap(
    data, row_labels, col_labels, ax=None, cbar_kw=None, cbarlabel="", **kwargs
):
    if ax is None:
        ax = plt.gca()

    if cbar_kw is None:
        cbar_kw = {}

    # Plot the heatmap
    im = ax.imshow(data, **kwargs)

    # Create colorbar
    cbar = ax.figure.colorbar(im, ax=ax, **cbar_kw)
    cbar.ax.set_ylabel(cbarlabel, rotation=-90, va="bottom")

    # Show all ticks and label them with the respective list entries.
    ax.set_xticks(np.arange(data.shape[1]), labels=col_labels)
    ax.set_yticks(np.arange(data.shape[0]), labels=row_labels)

    # Let the horizontal axes labeling appear on top.
    ax.tick_params(top=True, bottom=False, labeltop=True, labelbottom=False)

    # Rotate the tick labels and set their alignment.
    plt.setp(ax.get_xticklabels(), rotation=-30, ha="right", rotation_mode="anchor")

    # Turn spines off and create white grid.
    ax.spines[:].set_visible(False)

    ax.set_xticks(np.arange(data.shape[1] + 1) - 0.5, minor=True)
    ax.set_yticks(np.arange(data.shape[0] + 1) - 0.5, minor=True)
    ax.grid(which="minor", color="w", linestyle="-", linewidth=3)
    ax.tick_params(which="minor", bottom=False, left=False)

    return im, cbar

def annotate_heatmap(
    im,
    data=None,
    valfmt="{x:.2f}",
    textcolors=("black", "white"),
    threshold=None,
    **textkw,
):
    if not isinstance(data, (list, np.ndarray)):
        data = im.get_array()

    # Normalize the threshold to the images color range.
    if threshold is not None:
        threshold = im.norm(threshold)
    else:
        threshold = im.norm(data.max()) / 2.0

    # Set default alignment to center, but allow it to be
    # overwritten by textkw.
    kw = dict(horizontalalignment="center", verticalalignment="center")
    kw.update(textkw)

    # Get the formatter in case a string is supplied
    if isinstance(valfmt, str):
        valfmt = matplotlib.ticker.StrMethodFormatter(valfmt)

    # Loop over the data and create a `Text` for each "pixel".
    # Change the text's color depending on the data.
    texts = []
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            kw.update(color=textcolors[int(im.norm(data[i, j]) > threshold)])
            text = im.axes.text(j, i, valfmt(data[i, j], None), **kw)
            texts.append(text)

    return texts

fig, ((ax, ax2), (ax3, ax4)) = plt.subplots(2, 2, figsize=(10, 8))
fig.suptitle(suptitle, fontsize=16)

# Heatmap for vehicles count
im, _ = heatmap(
    vehicles_count, cities, transportation, ax=ax, cmap="Blues", cbarlabel=cbarlabels[0]
)
annotate_heatmap(im, valfmt="{x:.0f}", size=9)
ax.set_title("Vehicles Count")

# Heatmap for yearly transport count
yearly_data = np.random.randint(1000, 10000, size=(len(cities), len(transportation)))
im, _ = heatmap(yearly_data, cities, transportation, ax=ax2, cmap="Greens", cbarlabel=cbarlabels[1])
annotate_heatmap(im, valfmt="{x:.0f}", size=9)
ax2.set_title("Yearly Transport Count")

# Heatmap for efficiency rate
efficiency_data = np.random.rand(len(cities), len(transportation))
im, _ = heatmap(efficiency_data, cities, transportation, ax=ax3, cmap="Oranges", cbarlabel=cbarlabels[2])
annotate_heatmap(im, valfmt="{x:.2f}", size=9)
ax3.set_title("Efficiency Rate")

# Correlation heatmap
corr_matrix = np.corrcoef(vehicles_count.T)
im, _ = heatmap(
    corr_matrix,
    transportation,
    transportation,
    ax=ax4,
    cmap="coolwarm",
    vmin=-1,
    vmax=1,
    cbarlabel="Correlation coefficient",
)

def func(x, pos):
    return f"{x:.2f}".replace("0.", ".").replace("1.00", "")

annotate_heatmap(im, valfmt=matplotlib.ticker.FuncFormatter(func), size=7)
ax4.set_title("Correlation Matrix of Vehicle Counts")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout(rect=[0, 0, 1, 0.96])
plt.savefig("heatmap_103.pdf", bbox_inches="tight")
