
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib
import matplotlib as mpl

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
teams = ["Team A", "Team B", "Team C", "Team D", "Team E", "Team F", "Team G"]
seasons = ["Season 1", "Season 2", "Season 3", "Season 4", "Season 5", "Season 6", "Season 7"]

performance = np.array(
    [
        [78, 85, 92, 88, 75, 90, 80],
        [82, 78, 90, 84, 82, 79, 85],
        [75, 80, 89, 90, 83, 86, 88],
        [88, 87, 85, 89, 84, 82, 90],
        [86, 83, 81, 80, 87, 85, 92],
        [80, 78, 88, 87, 85, 92, 91],
        [85, 90, 78, 88, 86, 89, 84],
    ]
)

cbarlabels = ["Performance Score", "Games Played", "Correlation Coeff."]

# Text labels
xlabel = "Seasons"
ylabel = "Teams"
title = "Teams Performance Over Seasons"
supertitle = "Sports Performance Metrics"
legendlabels = ["Above Average", "Below Average"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

def heatmap(
    data, row_labels, col_labels, ax=None, cbar_kw=None, cbarlabel="", **kwargs
):
    if ax is None:
        ax = plt.gca()

    if cbar_kw is None:
        cbar_kw = {}

    # Plot the heatmap
    im = ax.imshow(data, **kwargs)

    # Create colorbar
    cbar = ax.figure.colorbar(im, ax=ax, **cbar_kw)
    cbar.ax.set_ylabel(cbarlabel, rotation=-90, va="bottom")

    # Show all ticks and label them with the respective list entries.
    ax.set_xticks(np.arange(data.shape[1]), labels=col_labels)
    ax.set_yticks(np.arange(data.shape[0]), labels=row_labels)

    # Let the horizontal axes labeling appear on top.
    ax.tick_params(top=True, bottom=False, labeltop=True, labelbottom=False)

    # Rotate the tick labels and set their alignment.
    plt.setp(ax.get_xticklabels(), rotation=-30, ha="right", rotation_mode="anchor")

    # Turn spines off and create white grid.
    ax.spines[:].set_visible(False)

    ax.set_xticks(np.arange(data.shape[1] + 1) - 0.5, minor=True)
    ax.set_yticks(np.arange(data.shape[0] + 1) - 0.5, minor=True)
    ax.grid(which="minor", color="w", linestyle="-", linewidth=3)
    ax.tick_params(which="minor", bottom=False, left=False)

    return im, cbar

def annotate_heatmap(
    im,
    data=None,
    valfmt="{x:.2f}",
    textcolors=("black", "white"),
    threshold=None,
    **textkw,
):
    if not isinstance(data, (list, np.ndarray)):
        data = im.get_array()

    # Normalize the threshold to the image color range.
    if threshold is not None:
        threshold = im.norm(threshold)
    else:
        threshold = im.norm(data.max()) / 2.0

    # Set default alignment to center, but allow it to be
    # overwritten by textkw.
    kw = dict(horizontalalignment="center", verticalalignment="center")
    kw.update(textkw)

    # Get the formatter in case a string is supplied
    if isinstance(valfmt, str):
        valfmt = matplotlib.ticker.StrMethodFormatter(valfmt)

    # Loop over the data and create a `Text` for each "pixel".
    # Change the text's color depending on the data.
    texts = []
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            kw.update(color=textcolors[int(im.norm(data[i, j]) > threshold)])
            text = im.axes.text(j, i, valfmt(data[i, j], None), **kw)
            texts.append(text)

    return texts

fig, ((ax, ax2), (ax3, ax4)) = plt.subplots(2, 2, figsize=(12, 10))
plt.suptitle(supertitle, fontsize=16)

# Replicate the above example with a different font size and colormap.
im, _ = heatmap(
    performance, teams, seasons, ax=ax, cmap="YlGnBu", cbarlabel=cbarlabels[0]
)
annotate_heatmap(im, valfmt="{x:d}", size=8)

# Create some new data, give further arguments to imshow
data = np.random.randint(10, 30, size=(7, 7))
matchups_y = [f"Matchup {i}" for i in range(1, 8)]
matchups_x = [f"Game {i}" for i in list("ABCDEFG")]
im, _ = heatmap(data, matchups_y, matchups_x, ax=ax2, vmin=0, cmap="RdYlBu", cbarlabel=cbarlabels[1])
annotate_heatmap(im, valfmt="{x:d}", size=8, threshold=15, textcolors=("black", "white"))

# Sometimes even the data itself is categorical. Here we use a
# `matplotlib.colors.BoundaryNorm` to get the data into classes
# and use this to colorize the plot, but also to obtain the class
# labels from an array of classes.
data = np.random.randn(6, 6)
labels_y = [f"Player {i}" for i in range(1, 7)]
labels_x = [f"Metric {i}" for i in range(1, 7)]

qrates = list("ABCDEFG")
norm = matplotlib.colors.BoundaryNorm(np.linspace(-3.5, 3.5, 8), 7)
fmt = matplotlib.ticker.FuncFormatter(lambda x, pos: qrates[::-1][norm(x)])

im, _ = heatmap(
    data,
    labels_y,
    labels_x,
    ax=ax3,
    cmap=mpl.colormaps["coolwarm"].resampled(7),
    norm=norm,
    cbar_kw=dict(ticks=np.arange(-3, 4), format=fmt),
    cbarlabel="Performance Rating",
)
annotate_heatmap(
    im, valfmt=fmt, size=9, fontweight="bold", threshold=0, textcolors=("black", "white")
)

# We can nicely plot a correlation matrix. Since this is bound by -1 and 1,
# we use those as vmin and vmax. We may also remove leading zeros and hide
# the diagonal elements (which are all 1) by using a
# `matplotlib.ticker.FuncFormatter`.
corr_matrix = np.corrcoef(performance)
im, _ = heatmap(
    corr_matrix,
    teams,
    teams,
    ax=ax4,
    cmap="coolwarm",
    vmin=-1,
    vmax=1,
    cbarlabel=cbarlabels[2],
)

def func(x, pos):
    return f"{x:.2f}".replace("0.", ".").replace("1.00", "")

annotate_heatmap(im, valfmt=matplotlib.ticker.FuncFormatter(func), size=8)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("heatmap_105.pdf", bbox_inches="tight")
