
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Define the data for average monthly temperatures (°C)
data = np.array(
    [
        [30, 25, 28, 33, 29, 31],
        [22, 19, 21, 26, 23, 27],
        [15, 17, np.nan, 20, 18, 19],  # Assumed another NaN value for padding
        [10, 12, 14, 13, 16, 15],
    ]
)

title = "Average Monthly Temperatures (°C)"
xlabel = "Month"
xticklabels = ["Jan", "Feb", "Mar", "Apr", "May", "Jun"]
yticklabels = ["City A", "City B", "City C", "City D"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting the heatmap with adjusted colorbar and new theme color
# Create mask for NaN values to hatch them later
mask = np.isnan(data)

# Defining a new color palette
cmap = plt.get_cmap("coolwarm")
norm = plt.Normalize(vmin=np.nanmin(data), vmax=np.nanmax(data))

fig, ax = plt.subplots(figsize=(10, 8))
cax = ax.imshow(data, cmap=cmap, norm=norm)
cbar = fig.colorbar(cax, ax=ax, extend="both")

# Add hatches for NaN values
for i, j in zip(*np.where(mask)):
    ax.add_patch(
        plt.Rectangle(
            (j - 0.5, i - 0.5), 1, 1, fill=False, hatch="//", edgecolor="black"
        )
    )

# Adding titles and labels
plt.title(title)
plt.xlabel(xlabel)

# Define the labels for x and y axis
ax.set_xticks(range(6))
ax.set_xticklabels(xticklabels, rotation=45)
ax.set_yticks(range(4))
ax.set_yticklabels(yticklabels, rotation=0)

# Add annotations
for i in range(4):
    for j in range(6):
        if not np.isnan(data[i, j]):
            if data[i, j] > np.nanmean(data) * 1.5:
                ax.text(
                    j, i, f"{data[i, j]:.0f}", ha="center", va="center", color="white"
                )
            else:
                ax.text(
                    j, i, f"{data[i, j]:.0f}", ha="center", va="center", color="black"
                )

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_46.pdf", bbox_inches="tight")
