
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import pandas as pd

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)

# Generating a similar style data for demonstration
# Assuming there are 3 seasons and 2 shows
seasons = ["Spring", "Summer", "Fall"]
shows = ["Show A", "Show B"]
media_data = []

# Generating random viewer ratings and ad revenue for the example
for show in shows:
    for season in seasons:
        ratings = np.random.randint(1, 10, size=9)  # Viewer ratings from 1 to 10
        ad_revenue = np.round(np.random.uniform(1000, 5000), 2)
        for rank, rating in enumerate(ratings, start=1):
            media_data.append(
                {
                    "Show": show,
                    "Season": season,
                    "Rank": rank,
                    "ViewerRating": rating,
                    "AdRevenue(K)": ad_revenue,
                }
            )

# Create the DataFrame
df = pd.DataFrame(media_data)

# Pivoting the dataframe for the heatmap
df_pivot = df.pivot_table(
    index="Rank", columns=["Show", "Season"], values="ViewerRating", aggfunc="first"
)

# Chart label parameters
title = "Media Show Performance Heatmap"
xlabel = "Show and Season"
ylabel = "Rating Rank"
color_bar_label = "Viewer Rating"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Initialize the matplotlib figure
fig, ax = plt.subplots(figsize=(10, 10))

# Create a custom colormap for the heatmap
cmap = plt.get_cmap("viridis")

# Draw the heatmap
cax = ax.imshow(df_pivot, cmap=cmap)

# Decorations
plt.title(title, fontsize=14)
plt.xlabel(xlabel, fontsize=12)
plt.ylabel(ylabel, fontsize=15)

# Set x-axis and y-axis labels
ax.set_xticks(range(len(df_pivot.columns)))
ax.set_xticklabels(
    [f"{col[0]} {col[1]}" for col in df_pivot.columns], rotation=45, ha="right"
)
ax.set_yticks(range(len(df_pivot.index)))
ax.set_yticklabels(df_pivot.index)

# Add annotations
for i in range(len(df_pivot.index)):
    for j in range(len(df_pivot.columns)):
        ax.text(j, i, df_pivot.iloc[i, j], ha="center", va="center", color="white")

# Add colorbar to be smaller
cbar = fig.colorbar(cax, ax=ax, fraction=0.2, pad=0.04)
cbar.set_label(color_bar_label)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_77.pdf", bbox_inches="tight")

