
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import pandas as pd

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generating a similar style data for demonstration
# Assuming there are 3 battles and 3 participants
battles = ["Battle of Gettysburg", "Battle of Hastings", "Battle of Waterloo"]
participants = ["Union", "Confederate", "Normans"]

casualty_data = []

# Generating random casualties and percentage changes for the example
for battle in battles:
    for participant in participants:
        casualties = np.random.randint(100, 1000, size=9)
        np.random.shuffle(casualties)
        percent_change = np.round(np.random.uniform(-30, 30), 2)
        for day in range(1, 10):
            casualty_data.append(
                {
                    "Battle": battle,
                    "Participant": participant,
                    "Day": day,
                    "Casualties": casualties[day-1],
                    "PercentChange": percent_change,
                }
            )

# Create the DataFrame
df = pd.DataFrame(casualty_data)

# Pivoting the dataframe for the heatmap
df_pivot = df.pivot_table(
    index="Day", columns=["Battle", "Participant"], values="Casualties", aggfunc="first"
)

title = "Historical Battle Casualties Heatmap"
xlabel = "Battle and Participant"
ylabel = "Day"
color_bar_label = "Number of Casualties"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Initialize the matplotlib figure
fig, ax = plt.subplots(figsize=(10, 10))

# Create a custom colormap for the heatmap
cmap = plt.get_cmap("YlOrBr")

# Draw the heatmap
cax = ax.imshow(df_pivot, cmap=cmap)

# Decorations
plt.title(title, fontsize=16)
plt.xlabel(xlabel, fontsize=14)
plt.ylabel(ylabel, fontsize=14)

# Set x-axis and y-axis labels
ax.set_xticks(range(len(df_pivot.columns)))
ax.set_xticklabels(
    [f"{col[0]} - {col[1]}" for col in df_pivot.columns], rotation=45, ha="right"
)
ax.set_yticks(range(len(df_pivot.index)))
ax.set_yticklabels(df_pivot.index)

# Add annotations
for i in range(len(df_pivot.index)):
    for j in range(len(df_pivot.columns)):
        ax.text(j, i, df_pivot.iloc[i, j], ha="center", va="center", color="black")

# Add colorbar to be smaller
cbar = fig.colorbar(cax, ax=ax, fraction=0.02, pad=0.02)
cbar.set_label(color_bar_label)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_78.pdf", bbox_inches="tight")
