
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Defining the colormap from white to green-blue
cmap = plt.cm.cubehelix

# Data for the two subplots (Climate metrics across different regions)
data1 = np.array(
    [
        [22.5, 120.0, 0.7, 78.0, 1.4, 60.3],
        [25.2, 100.2, 0.6, 65.8, 3.3, 70.5],
        [20.8, 130.3, 0.8, 80.2, 2.5, 75.0],
        [30.0, 95.1, 0.5, 50.1, 4.1, 40.2],
        [15.7, 140.8, 1.2, 90.5, 1.1, 25.5],
    ]
)

data2 = np.array(
    [
        [18.3, 160.0, 1.0, 110.4, 3.2, 50.0],
        [22.4, 115.5, 0.9, 85.0, 2.8, 80.0],
        [19.6, 150.5, 0.7, 120.3, 2.0, 60.0],
        [21.5, 105.2, 0.8, 98.5, 3.6, 70.0],
        [18.0, 130.0, 1.2, 95.0, 4.0, 65.0],
    ]
)

# X and Y labels
x_labels = ["Temperature", "Precipitation", "Wind Speed", "Humidity", "CO2 Levels", "Sunshine Hours"]
y_labels = ["Region A", "Region B", "Region C", "Region D", "Region E"]

# Subplot titles
titles = ["(a) Climate Metrics in January", "(b) Climate Metrics in July"]

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=0.1, vmax=200)

# Axes Limits and Labels
xticks_values = range(len(x_labels))
yticks_values = range(len(y_labels))
colorbar_ticks = [0.1, 1, 10, 100, 200]
yticklabels = ["0.1", "1", "10", "100", "200"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure
fig, axes = plt.subplots(
    1, 2, figsize=(20, 8), gridspec_kw={"width_ratios": [1, 1], "wspace": 0.3}
)


# Function to create a subplot
def create_subplot(ax, data, title):
    # Create the scatter plot
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            # Calculate the color based on the original value
            color = cmap(norm(data[i, j]))

            # Draw the circle with a fixed size
            circle = plt.Circle((j, i), 0.4, color=color)  # Fixed size
            ax.add_artist(circle)

            # Determine text color based on the value
            text_color = "white" if data[i, j] > 50 else "black"

            # Add the text inside the circle
            ax.text(
                j, i, f"{data[i, j]:.1f}", ha="center", va="center", color=text_color
            )

    # Set labels for x and y axes
    ax.set_xticks(range(len(x_labels)))
    ax.set_xticklabels(x_labels, ha="center")
    ax.set_yticks(range(len(y_labels)))
    ax.set_yticklabels(y_labels, va="center")

    # Adding the title for the subplot
    ax.set_title(title, fontsize=16)

    # Set the limits of the axes; they should be one more than your data range
    ax.set_xlim(-0.5, data.shape[1] - 0.5)
    ax.set_ylim(-0.5, data.shape[0] - 0.5)

    # Set the aspect of the plot to be equal and add a frame
    ax.set_aspect("equal")
    for spine in ax.spines.values():
        spine.set_visible(True)


# Create each subplot
create_subplot(axes[0], data1, titles[0])
create_subplot(axes[1], data2, titles[1])

# Create a colorbar on the far right side of the figure
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = fig.colorbar(
    sm,
    ax=axes,
    ticks=colorbar_ticks,
    orientation="vertical",
    fraction=0.015,
    pad=0.05,
)
cbar.ax.set_yticklabels(yticklabels)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_85.pdf", bbox_inches="tight")
