
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)

# Defining the colormap from blue to red
cmap = plt.cm.coolwarm

# Data for the two subplots
data1 = np.array(
    [
        [15.3, 18.0, 20.9, 25.0, 30.4, 35.3],
        [14.2, 17.2, 21.6, 26.8, 31.3, 36.9],
        [13.8, 18.3, 22.9, 27.2, 32.5, 37.3],
        [10.0, 12.1, 18.5, 24.1, 29.1, 34.2],
        [5.7, 8.1, 15.6, 20.5, 24.6, 29.5],
    ]
)

data2 = np.array(
    [
        [25.5, 30.0, 35.7, 40.9, 45.0, 50.0],
        [24.4, 29.0, 34.8, 39.9, 44.0, 48.0],
        [22.6, 27.2, 32.8, 37.3, 41.1, 45.0],
        [20.3, 25.2, 30.0, 35.5, 40.0, 44.8],
        [15.5, 20.3, 25.4, 30.5, 35.0, 39.3],
    ]
)

# X and Y labels
x_labels = ["Morning", "Noon", "Afternoon", "Evening", "Night", "Midnight"]
y_labels = ["Sunny", "Cloudy", "Rainy", "Snowy", "Stormy"]

# Subplot titles
titles = ["(a) Temperature trends throughout the day", "(b) Average temperatures by weather condition"]

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=0.1, vmax=50)

# Axes Limits and Labels
xticks_values = range(len(x_labels))
yticks_values = range(len(y_labels))
colorbar_ticks = [0.1, 1, 10, 50]
yticklabels = ["0.1", "1", "10", "50"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure
fig, axes = plt.subplots(
    1, 2, figsize=(20, 8), gridspec_kw={"width_ratios": [1, 1], "wspace": 0.3}
)

# Function to create a subplot
def create_subplot(ax, data, title):
    # Create the scatter plot
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            # Calculate the color based on the original value
            color = cmap(norm(data[i, j]))

            # Draw the circle with a fixed size
            circle = plt.Circle((j, i), 0.4, color=color)  # Fixed size
            ax.add_artist(circle)

            # Determine text color based on the value
            text_color = "white" if data[i, j] > 25 else "black"

            # Add the text inside the circle
            ax.text(
                j, i, f"{data[i, j]:.1f}°C", ha="center", va="center", color=text_color
            )

    # Set labels for x and y axes
    ax.set_xticks(range(len(x_labels)))
    ax.set_xticklabels(x_labels, ha="center")
    ax.set_yticks(range(len(y_labels)))
    ax.set_yticklabels(y_labels, va="center")

    # Adding the title for the subplot
    ax.set_title(title, fontsize=16)

    # Set the limits of the axes; they should be one more than your data range
    ax.set_xlim(-0.5, data.shape[1] - 0.5)
    ax.set_ylim(-0.5, data.shape[0] - 0.5)

    # Set the aspect of the plot to be equal and add a frame
    ax.set_aspect("equal")
    for spine in ax.spines.values():
        spine.set_visible(True)


# Create each subplot
create_subplot(axes[0], data1, titles[0])
create_subplot(axes[1], data2, titles[1])

# Create a colorbar on the far right side of the figure
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = fig.colorbar(
    sm,
    ax=axes,
    ticks=colorbar_ticks,
    orientation="vertical",
    fraction=0.015,
    pad=0.05,
)
cbar.ax.set_yticklabels(yticklabels)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_87.pdf", bbox_inches="tight")
