
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Custom colormap from light green to dark green
cmap = plt.cm.Greens

# Sample data: Population distribution (in thousands) for different age groups and regions
data = np.array(
    [
        [100.3, 200.0, 150.0, 300.0, 400.0, 250.0],
        [120.0, 220.0, 180.0, 320.0, 450.0, 280.0],
        [110.5, 210.0, 160.0, 310.0, 430.0, 260.0],
        [90.0, 190.0, 140.0, 290.0, 380.0, 240.0],
        [130.0, 230.0, 170.0, 330.0, 470.0, 290.0],
    ]
)

# X and Y labels
xticklabels = ["0-18", "19-35", "36-50", "51-65", "66-80", "81+"]
yticklabels = ["Region A", "Region B", "Region C", "Region D", "Region E"]
x_label = "Age Group"
y_label = "Region"
title = "Population Distribution by Age Group and Region"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure and axis
fig, ax = plt.subplots(figsize=(10, 8))

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=50, vmax=500)

# Create the scatter plot
for i in range(data.shape[0]):
    for j in range(data.shape[1]):
        # Calculate the color based on the original value
        color = cmap(norm(data[i, j]))

        # Draw the circle
        circle = plt.Circle((j, i), 0.5, color=color)  # Adjusted size
        ax.add_artist(circle)

        # Determine text color based on the value
        text_color = "white" if data[i, j] > 300 else "black"

        # Add the text inside the circle
        ax.text(j, i, f"{data[i, j]:.1f}k", ha="center", va="center", color=text_color)

# Set labels for x and y axes
ax.set_xticks(range(len(xticklabels)))
ax.set_xticklabels(xticklabels, ha="center")
ax.set_yticks(range(len(yticklabels)))
ax.set_yticklabels(yticklabels, va="center")

# Adding titles for the axes and the plot
ax.set_xlabel(x_label, fontsize=14)
ax.set_ylabel(y_label, fontsize=14)
ax.set_title(title, fontsize=16, pad=20)

# Set the limits of the axes; they should be one more than your data range
ax.set_xlim(-0.5, data.shape[1] - 0.5)
ax.set_ylim(-0.5, data.shape[0] - 0.5)

# Set the aspect of the plot to be equal and add a frame
ax.set_aspect("equal")
for spine in ax.spines.values():
    spine.set_visible(True)

# Create a colorbar
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = plt.colorbar(sm, ax=ax, ticks=[50, 100, 200, 500], orientation="vertical")
cbar.ax.set_yticklabels(["50", "100", "200", "500"])

# Add gridlines
plt.grid(True, which="both", color="gray", linestyle="--", linewidth=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to fit the figure size
plt.tight_layout()

# Save the plot
plt.savefig("heatmap_92.pdf", bbox_inches="tight")
