
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate bimodal data for stress in young adults
young_adult_peak1 = np.random.normal(loc=25, scale=3, size=200)
young_adult_peak2 = np.random.normal(loc=35, scale=3, size=200)
young_adult_stress_levels = np.concatenate([young_adult_peak1, young_adult_peak2])

# Generate bimodal data for stress in midlife adults
midlife_peak1 = np.random.normal(loc=45, scale=4, size=500)
midlife_peak2 = np.random.normal(loc=60, scale=4, size=500)
midlife_stress_levels = np.concatenate([midlife_peak1, midlife_peak2])

labels = ["Young Adults", "Midlife Adults"]
xticks = np.arange(20, 70, 10)
yticks = [1, 10, 100]
xlabel = "Age"
ylabel = "Frequency"
title = "Stress Levels across Different Age Groups"
legendtitle = "Age Groups"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
plt.figure(figsize=(8, 6))

# Define the number of bins and bin edges for consistent bin width
bins = np.histogram(np.hstack((young_adult_stress_levels, midlife_stress_levels)), bins=20)[1]

# Create the histograms without stacking
# New colors using pastel hex codes
plt.hist(
    young_adult_stress_levels,
    bins=bins,
    color="#87CEEB",
    label=labels[0],
    edgecolor="white",
    linewidth=1,
    alpha=0.7,
    zorder=2,
)
plt.hist(
    midlife_stress_levels,
    bins=bins,
    color="#FFB6C1",
    label=labels[1],
    edgecolor="white",
    linewidth=1,
    alpha=0.7,
    zorder=3,
)

# Set the background color to a light grey
plt.gca().set_facecolor("#f0f0f0")

# Set the scale of y-axis to logarithmic
plt.yscale("log")

# Set the x-axis ticks
plt.xticks(xticks)
plt.tick_params(axis="x", length=0)

# Add white grid lines and place them behind the bars (zorder=0)
plt.grid(color="white", linestyle="-", linewidth=1.5, zorder=0)

# Set the y-axis ticks and remove all line markings (spines)
plt.yticks(yticks)
plt.tick_params(axis="y", length=0)

# Remove spine lines
for spine in plt.gca().spines.values():
    spine.set_visible(False)

# Remove minor ticks on the y-axis
plt.tick_params(axis="y", which="minor", length=0)

# Set labels and title
plt.xlabel(xlabel)
plt.ylabel(ylabel)
plt.title(title)

# Move legend to the bottom center of the plot
plt.legend(
    title=legendtitle, loc="upper center", bbox_to_anchor=(0.5, -0.15), ncol=2
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to make room for the legend
plt.tight_layout()

plt.savefig("hist_59.pdf", bbox_inches="tight")
