
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D  # Importing Line2D for creating custom legend items

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data: Environmental Science Context - Air Quality Monitoring
sampling_points = [1, 2, 3, 4, 5, 6, 7]
pm25_levels = [35.2, 33.1, 30.5, 28.4, 26.3, 24.8, 23.5]
no2_levels = [50.5, 48.2, 45.6, 43.1, 40.8, 38.9, 37.4]
avg_pm25_levels = [30] * len(sampling_points)
avg_no2_levels = [45] * len(sampling_points)

# Axes Limits and Labels
xlabel_value = "Number of Sampling Points"
ylabel_value_1 = "PM2.5 Levels (µg/m³)"
ylabel_value_2 = "NO2 Levels (ppb)"
ylim_values_1 = [20, 40]
ylim_values_2 = [35, 55]
yticks_values_1 = range(20, 41, 5)
yticks_values_2 = range(35, 56, 5)

# Labels
label_pm25 = "PM2.5 Levels"
label_no2 = "NO2 Levels"
avg_label_pm25 = "Avg PM2.5"
avg_label_no2 = "Avg NO2"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plot
fig, ax1 = plt.subplots(figsize=(10, 6))

# PM2.5 plot
(pm25_line,) = ax1.plot(
    sampling_points,
    pm25_levels,
    "o-",
    color="#66c2a5",  # Soft green
    label=label_pm25,
    markersize=10,
    linewidth=3,
    mec="black",
)
ax1.set_xlabel(xlabel_value, fontsize=14)
ax1.set_ylabel(ylabel_value_1, fontsize=14, color="#66c2a5")
ax1.tick_params(
    axis="y", labelcolor="#66c2a5", direction="in", rotation=90, labelsize=12
)
ax1.tick_params(
    axis="x",
    direction="in",
    labelsize=12,
)
ax1.set_yticks(yticks_values_1)
ax1.set_ylim(ylim_values_1)

# Adding PM2.5 values to the plot
for i, txt in enumerate(pm25_levels):
    ax1.annotate(
        f"{txt} µg/m³",
        (sampling_points[i], txt),
        textcoords="offset points",
        xytext=(0, 10),
        ha="center",
        fontsize=12,
    )

# NO2 plot with a secondary y-axis
ax2 = ax1.twinx()
(no2_line,) = ax2.plot(
    sampling_points,
    no2_levels,
    "s-",
    color="#fc8d62",  # Soft brown
    label=label_no2,
    markersize=10,
    linewidth=3,
    mec="black",
)
ax2.set_ylabel(ylabel_value_2, color="#fc8d62", fontsize=14)
ax2.tick_params(
    axis="y", labelcolor="#fc8d62", direction="in", rotation=90, labelsize=12
)
ax2.set_yticks(yticks_values_2)
ax2.set_ylim(ylim_values_2)

# Adding NO2 values to the plot
for i, txt in enumerate(no2_levels):
    ax2.annotate(
        f"{txt} ppb",
        (sampling_points[i], txt),
        textcoords="offset points",
        xytext=(0, -20),
        ha="center",
        fontsize=12,
    )

# Avg PM2.5 and NO2 dashed lines
ax1.axhline(y=30, color="#66c2a5", linestyle=":", linewidth=2)
ax2.axhline(y=45, color="#fc8d62", linestyle="--", linewidth=2)

# Creating custom legend items
avg_pm25_legend = Line2D([0], [0], color="#66c2a5", linestyle=":", linewidth=2, label=avg_label_pm25)
avg_no2_legend = Line2D([0], [0], color="#fc8d62", linestyle="--", linewidth=2, label=avg_label_no2)

# Adding legends
first_legend = ax1.legend(
    handles=[avg_pm25_legend, avg_no2_legend],
    loc="upper left",
    ncol=2,
    fontsize=14,
    edgecolor="black",
)
ax1.add_artist(first_legend)  # Add the first legend manually
second_legend = ax1.legend(
    handles=[pm25_line, no2_line], loc="upper right", fontsize=14, edgecolor="black"
)  # Add the second legend

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("line_172.pdf", bbox_inches="tight")
