
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)

# Updated steps
days = np.linspace(0, 365, 50)

# Generating distinct trends for each line
temperature = (np.sin(days * 0.017) * 10) + 20  # Sinusoidal trend for temperature
humidity = np.array(days) ** 2 * 0.0001 + 60  # Quadratic growth for humidity
wind_speed = np.random.normal(loc=5, scale=1, size=len(days))  # Random noise for wind speed
precipitation = np.exp(-0.01 * days) * 100  # Exponential decay for precipitation

# Simulating standard deviations for error
std_dev_temp = 2.0
std_dev_humid = 5.0
std_dev_wind = 1.0
std_dev_precip = 10.0

# Data standard deviations
temperature_std = np.full_like(temperature, std_dev_temp)
humidity_std = np.full_like(humidity, std_dev_humid)
wind_speed_std = np.full_like(wind_speed, std_dev_wind)
precipitation_std = np.full_like(precipitation, std_dev_precip)

# Axes Limits and Labels
xlabel_value = "Days"
xlim_values = [0, 365]
xticks_values = np.linspace(0, 365, 13)

ylabel_value_1 = "Temperature (°C) / Humidity (%)"
ylabel_value_2 = "Wind Speed (km/h) / Precipitation (mm)"
yticks_values_1 = np.arange(0, 100, 10)
yticks_values_2 = np.arange(0, 120, 10)
ylim_values_1 = [0, 100]
ylim_values_2 = [0, 120]

# Labels
label_1 = "Temperature (°C)"
label_2 = "Humidity (%)"
label_3 = "Wind Speed (km/h)"
label_4 = "Precipitation (mm)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating a figure with two subplots
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(10, 12))

# Plotting on the first subplot
ax1.plot(days, temperature, "o-", color="#FFA500", label=label_1)
ax1.fill_between(days, temperature - temperature_std, temperature + temperature_std, color="#FFA500", alpha=0.2)

ax1.plot(days, humidity, "s-", color="#1E90FF", label=label_2)
ax1.fill_between(days, humidity - humidity_std, humidity + humidity_std, color="#1E90FF", alpha=0.2)

# Plotting on the second subplot
ax2.plot(days, wind_speed, "^--", color="#708090", markerfacecolor=(0, 0, 0, 0), markeredgecolor="#708090", label=label_3)
ax2.fill_between(days, wind_speed - wind_speed_std, wind_speed + wind_speed_std, color="#708090", alpha=0.2)

ax2.plot(days, precipitation, "*-", color="#32CD32", label=label_4)
ax2.fill_between(days, precipitation - precipitation_std, precipitation + precipitation_std, color="#32CD32", alpha=0.2)

# Customizing the first subplot
ax1.set_xlabel(xlabel_value, fontsize=12)
ax1.set_ylabel(ylabel_value_1, fontsize=12)
ax1.set_xticks(xticks_values)
ax1.set_yticks(yticks_values_1)
ax1.set_xlim(xlim_values)
ax1.set_ylim(ylim_values_1)
ax1.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax1.grid(True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5)
ax1.set_facecolor("#f9f9f9")

# Customizing the second subplot
ax2.set_xlabel(xlabel_value, fontsize=12)
ax2.set_ylabel(ylabel_value_2, fontsize=12)
ax2.set_xticks(xticks_values)
ax2.set_yticks(yticks_values_2)
ax2.set_xlim(xlim_values)
ax2.set_ylim(ylim_values_2)
ax2.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax2.grid(True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5)
ax2.set_facecolor("#f9f9f9")

# Tight layout and saving output
plt.tight_layout()
plt.savefig("line_191.pdf", bbox_inches="tight")
