
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Generate new data representing energy metrics over time
months = np.linspace(1, 12, 12)  # 12 months

# Simulate Renewable and Non-Renewable Energy Trends
renewable_energy = 0.4 + 0.05 * months + 0.1 * np.sin(2 * np.pi * months / 12)
non_renewable_energy = 0.8 - 0.03 * months + 0.1 * np.cos(2 * np.pi * months / 12)

# Calculate uncertainty bounds for visual emphasis
upper_bound_renewable = renewable_energy + 0.05
lower_bound_renewable = renewable_energy - 0.05
upper_bound_non_renewable = non_renewable_energy + 0.05
lower_bound_non_renewable = non_renewable_energy - 0.05

# Axes Limits and Labels
xlabel_value = "Time (Months)"
xlim_values = [1, 12]
xticks_values = np.arange(1, 13, 1)

ylabel_value = "Energy Usage (Normalized)"
ylim_values = [0, 1.1]
yticks_values = np.linspace(0, 1, 6)

# Labels
label_1 = "Renewable Energy Usage"
label_2 = "Non-Renewable Energy Consumption"

# Title
title = "Energy Usage Trends Over Time"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size and create a single plot
fig, ax = plt.subplots(figsize=(10, 6))

# Plot renewable energy usage with filled uncertainty
ax.fill_between(
    months,
    lower_bound_renewable,
    upper_bound_renewable,
    color="#98FB98",  # Light green
    alpha=0.3,
)
ax.plot(
    months,
    renewable_energy,
    label=label_1,
    color="#006400",  # Dark green
    linestyle="-",
    linewidth=2,
    marker='o'
)

# Plot non-renewable energy consumption with filled uncertainty
ax.fill_between(
    months,
    lower_bound_non_renewable,
    upper_bound_non_renewable,
    color="#D3D3D3",  # Light grey
    alpha=0.3,
)
ax.plot(
    months,
    non_renewable_energy,
    label=label_2,
    color="#696969",  # Dim grey
    linestyle="--",
    linewidth=2,
    marker='s'
)

# Customize the axes and grid
ax.set_title(title)
ax.set_xlabel(xlabel_value, fontsize=14)
ax.set_ylabel(ylabel_value, fontsize=14)
ax.set_xticks(xticks_values)
ax.set_yticks(yticks_values)
ax.grid(True, which='both', linestyle='--', linewidth=0.5)

# Add a legend to the plot
ax.legend(loc="upper right", frameon=True, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Enhance layout and display
plt.tight_layout()
plt.savefig("line_209.pdf", bbox_inches="tight")
