
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate new data representing seasonal temperature variations and precipitation levels over time
time_points = np.linspace(0, 12, 100)  # Representing months in a year

# Simulate temperature variations (sinusoidal pattern)
temperature = 10 * np.sin(2 * np.pi * time_points / 12) + 20

# Simulate precipitation levels with a random seasonal trend
precipitation = np.random.uniform(0, 5, size=time_points.shape) + 3 + 2 * np.cos(2 * np.pi * time_points / 12)

# Calculate uncertainty bounds for visual emphasis
upper_bound_temperature = temperature + 2
lower_bound_temperature = temperature - 2
upper_bound_precipitation = precipitation + 1
lower_bound_precipitation = precipitation - 1

# Axes Limits and Labels
xlabel_value = "Time (Months)"
xlim_values = [0, 12]
xticks_values = np.arange(0, 13, 1)

ylabel_value = "Value"
ylim_values = [0, 34]
yticks_values = np.linspace(0, 34, 7)

# Labels
label_temperature = "Temperature (°C)"
label_precipitation = "Precipitation (mm)"

# Title
title = "Seasonal Variations in Temperature and Precipitation"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size and create a single plot
fig, ax = plt.subplots(figsize=(10, 6))

# Plot the temperature variations with filled uncertainty
ax.fill_between(
    time_points,
    lower_bound_temperature,
    upper_bound_temperature,
    color="lightgreen",
    alpha=0.3,
)
ax.plot(
    time_points,
    temperature,
    label=label_temperature,
    color="green",
    linestyle="-",
    linewidth=2,
    marker='o',
)

# Plot the precipitation levels with filled uncertainty
ax.fill_between(
    time_points,
    lower_bound_precipitation,
    upper_bound_precipitation,
    color="lightblue",
    alpha=0.3,
)
ax.plot(
    time_points,
    precipitation,
    label=label_precipitation,
    color="blue",
    linestyle="--",
    linewidth=2,
    marker='x',
)

# Customize the axes and grid
ax.set_title(title, fontsize=16)
ax.set_xlabel(xlabel_value, fontsize=14)
ax.set_ylabel(ylabel_value, fontsize=14)
ax.set_xlim(xlim_values)
ax.set_ylim(ylim_values)
ax.set_xticks(xticks_values)
ax.set_yticks(yticks_values)
ax.grid(True, linestyle='--', alpha=0.7)

# Add a legend to the plot
ax.legend(loc="upper right", frameon=True, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Enhance layout and display
plt.tight_layout()
plt.savefig("line_211.pdf", bbox_inches="tight")
