
# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)  # Different seed for new data generation

weeks = np.arange(1, 29)
stress_data = np.random.normal(loc=0.5, scale=0.1, size=len(weeks)).clip(0, 1)
happiness_data = np.random.normal(loc=0.6, scale=0.15, size=len(weeks)).clip(0, 1)
productivity_data = 1 - (0.5 * stress_data + 0.5 * happiness_data)

# Extracted variables
xlabel = "Weeks"
ylabel_stress = "Stress Level"
ylabel_happiness = "Happiness Level"
ylabel_productivity = "Productivity Level"
xlim = (0, 30)
ylim = (0, 1.2)
title = "Psychological Indicators Over Time"
line_label_stress = "Stress"
line_label_happiness = "Happiness"
line_label_productivity = "Productivity"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis
fig, ax1 = plt.subplots(figsize=(10, 6))

# Plot the stress data on the primary y-axis
color = "#4682B4"  # SteelBlue
ax1.set_xlabel(xlabel)
ax1.set_ylabel(ylabel_stress, color=color)
(line1,) = ax1.plot(
    weeks,
    stress_data,
    color=color,
    marker="o",
    linestyle="-",
    clip_on=False,
    zorder=10,
    linewidth=2,
    label=line_label_stress,
)
ax1.tick_params(axis="y", labelcolor=color)
ax1.set_ylim(ylim)

# Create a secondary y-axis for the happiness data
ax2 = ax1.twinx()
color = "#6A5ACD"  # SlateBlue
ax2.set_ylabel(ylabel_happiness, color=color)
(line2,) = ax2.plot(
    weeks,
    happiness_data,
    color=color,
    marker="s",
    linestyle="--",
    clip_on=False,
    zorder=10,
    linewidth=2,
    label=line_label_happiness,
)
ax2.tick_params(axis="y", labelcolor=color)
ax2.set_ylim(ylim)

# Add a third line for productivity data
color = "#32CD32"  # LimeGreen
(line3,) = ax1.plot(
    weeks,
    productivity_data,
    color=color,
    marker="^",
    linestyle=":",
    linewidth=2,
    clip_on=False,
    zorder=10,
    label=line_label_productivity,
)

# Customize the plot with a title, grid, and background color
fig.patch.set_facecolor("#f4f4f4")
ax1.set_facecolor("#e0f7fa")
ax2.set_facecolor("#fce4ec")
ax1.set_xlim(xlim)
ax1.tick_params(axis="both", which="both", length=0)
ax2.tick_params(axis="both", which="both", length=0)
lines = [line1, line2, line3]
labels = [line.get_label() for line in lines]
fig.legend(
    lines, labels, loc="upper center", bbox_to_anchor=(0.5, 0.95), ncol=3, frameon=False
)
# Set the title and display the plot
plt.title(title, y=1.05)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better spacing and display
plt.tight_layout()
plt.savefig("line_254.pdf", bbox_inches="tight")
