
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

weeks = list(range(1, 29))
exercise_hours = np.random.normal(loc=4, scale=1, size=28)
caloric_intake = np.random.normal(loc=2500, scale=200, size=28)
weight_loss_progress = np.linspace(0, 10, 28) + (np.random.randn(28) * 0.5)

# Extracted variables
xlabel = "Weeks"
ylabel_exercise = "Exercise Hours"
ylabel_calories = "Caloric Intake (kcal)"
ylabel_weight_loss = "Weight Loss (kg)"
xlim = (0, 30)
ylim_exercise = (0, 10)
ylim_calories = (1800, 3200)
ylim_weight_loss = (-1, 12)
title = "Health Metrics Over Time"
line_label_exercise = "Exercise Hours"
line_label_calories = "Caloric Intake"
line_label_weight_loss = "Weight Loss Progress"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis
fig, ax1 = plt.subplots(figsize=(10, 6))

# Plot the exercise hours data on the primary y-axis
color = "#2ca02c"  # Green color
ax1.set_xlabel(xlabel)
ax1.set_ylabel(ylabel_exercise, color=color)
(line1,) = ax1.plot(
    weeks,
    exercise_hours,
    color=color,
    marker="o",
    linestyle="-",
    clip_on=False,
    zorder=10,
    linewidth=2,
    label=line_label_exercise,
)
ax1.tick_params(axis="y", labelcolor=color)
ax1.set_ylim(ylim_exercise)

# Create a secondary y-axis for caloric intake data
ax2 = ax1.twinx()
color = "#ff7f0e"  # Orange color
ax2.set_ylabel(ylabel_calories, color=color)
(line2,) = ax2.plot(
    weeks,
    caloric_intake,
    color=color,
    marker="x",
    linestyle="--",
    clip_on=False,
    zorder=10,
    linewidth=2,
    label=line_label_calories,
)
ax2.tick_params(axis="y", labelcolor=color)
ax2.set_ylim(ylim_calories)

# Adding weight loss progress to the primary y-axis with secondary scale for clarity
ax3 = ax1.twinx()
# Offset the third y-axis
ax3.spines["right"].set_position(("outward", 60))
color = "#1f77b4"  # Blue color
ax3.set_ylabel(ylabel_weight_loss, color=color)
(line3,) = ax3.plot(
    weeks,
    weight_loss_progress,
    color=color,
    marker="^",
    linestyle=":",
    linewidth=2,
    clip_on=False,
    zorder=10,
    label=line_label_weight_loss,
)
ax3.tick_params(axis="y", labelcolor=color)
ax3.set_ylim(ylim_weight_loss)

# Add a legend to the plot
lines = [line1, line2, line3]
labels = [line.get_label() for line in lines]
fig.legend(
    lines, labels, loc="upper center", bbox_to_anchor=(0.5, 0.95), ncol=3, frameon=False
)

# Customize the plot with a title, grid, and background color
fig.patch.set_facecolor("#f4f4f4")
ax1.set_facecolor("#e5f5f9")
ax2.set_facecolor("#f9e5e6")
ax3.set_facecolor("#eaf5e6")
ax1.set_xlim(xlim)
ax1.grid(True, which='both', linestyle='--', linewidth=0.5)
ax1.tick_params(axis="both", which="both", length=0)
ax2.tick_params(axis="both", which="both", length=0)
ax3.tick_params(axis="both", which="both", length=0)
plt.title(title, y=1.05)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better spacing and display
plt.tight_layout()
plt.savefig("line_255.pdf", bbox_inches="tight")
