
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Data for the plot
years = np.arange(2010, 2020)
gdp_growth = np.clip(
    np.linspace(2, 5, len(years)) + np.random.normal(0, 0.5, len(years)), 1, 6
)
unemployment_rate = np.clip(
    np.linspace(10, 5, len(years)) + np.random.normal(0, 0.5, len(years)), 4, 11
)
inflation_rate = [2.2, 2.3, 2.5, 2.7, 2.8, 3.0, 3.1, 3.2, 3.4, 3.5]
interest_rate = [1.5, 1.7, 1.8, 2.0, 2.3, 2.5, 2.8, 3.0, 3.2, 3.5]

# Axes Limits and Labels
xlabel_value = "Year"
ylabel_value_1 = "GDP Growth (%)"
ylabel_value_2 = "Unemployment Rate (%)"
ylim_values_1 = [0, 7]
ylim_values_2 = [3, 12]
yticks_values_1 = np.arange(0, 8, 1)
yticks_values_2 = np.arange(3, 13, 1)

# Labels
label_1 = "GDP Growth"
label_2 = "Unemployment Rate"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis
fig, ax1 = plt.subplots(figsize=(10, 8))

# GDP Growth and Unemployment Rate plots
(gdp_line,) = ax1.plot(
    years,
    gdp_growth,
    color="#1f77b4",
    label=label_1,
    markersize=8,
    linewidth=3,
    mec="black",
    linestyle="--",
    marker="D",
)
ax1.set_xlabel(xlabel_value, fontsize=14)
ax1.set_ylabel(ylabel_value_1, fontsize=14, color="#1f77b4")
ax1.tick_params(
    axis="y", labelcolor="#1f77b4", direction="in", labelsize=12
)
ax1.tick_params(axis="x", direction="in", labelsize=12)
ax1.set_yticks(yticks_values_1)
ax1.set_ylim(ylim_values_1)

# Create a secondary y-axis for Unemployment Rate
ax2 = ax1.twinx()
(unemployment_line,) = ax2.plot(
    years,
    unemployment_rate,
    color="#ff7f0e",
    label=label_2,
    markersize=8,
    linewidth=3,
    mec="black",
    linestyle="-.",
    marker="o",
)
ax2.set_ylabel(ylabel_value_2, color="#ff7f0e", fontsize=14)
ax2.tick_params(
    axis="y", labelcolor="#ff7f0e", direction="in", labelsize=12
)
ax2.set_yticks(yticks_values_2)
ax2.set_ylim(ylim_values_2)

# Inflation and Interest Rates trends
ax1.plot(years, inflation_rate, ":", color="green", linewidth=2, label="Inflation Rate")
ax1.plot(years, interest_rate, "--", color="#9467bd", linewidth=2, label="Interest Rate")

# Custom legend for the plot
inflation_legend = Line2D(
    [0], [0], color="green", linestyle=":", linewidth=2, label="Inflation Rate"
)
interest_legend = Line2D(
    [0], [0], color="#9467bd", linestyle="--", linewidth=2, label="Interest Rate"
)

# Add the legend to the plot
first_legend = ax1.legend(
    handles=[inflation_legend, interest_legend],
    loc="upper left",
    ncol=2,
    fontsize=14,
    edgecolor="black",
)
ax1.add_artist(first_legend)  # Add the first legend back to the plot
second_legend = ax1.legend(
    handles=[gdp_line, unemployment_line], loc="upper right", fontsize=14, edgecolor="black"
)  # Add the second legend

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better spacing and display
plt.tight_layout()
plt.savefig("line_272.pdf", bbox_inches="tight")
