
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Data for the plot
months = np.array(range(1, 13))
online_sales = np.clip(
    np.sin(np.linspace(0, 2 * np.pi, len(months))) * 100
    + np.random.normal(0, 20, len(months))
    + 150,
    50,
    250,
)
store_sales = np.clip(
    np.cos(np.linspace(0, 3 * np.pi, len(months))) * 120
    + np.random.normal(0, 30, len(months))
    + 200,
    70,
    300,
)
highlight = [3, 7, 11]

# Axes Limits and Labels
xlabel_value = "Month"
ylabel_value_1 = "Online Sales (in thousands)"
ylabel_value_2 = "Store Sales (in thousands)"

# Labels
label_1 = "Store Sales"
label_2 = "Monthly Target"

label = "Online Sales"

# Titles for the charts
title_1 = "Monthly Online Sales Performance"
title_2 = "Monthly Store Sales Performance"

# Supertitle for the entire figure
supertitle = "Sales Performance Over a Year"

# Additional labels for annotation
highlight_annotation = "Peak Month"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with a 1x2 grid
fig, axs = plt.subplots(1, 2, figsize=(14, 6))

# Plot the Online Sales data
(online_sales_line,) = axs[0].plot(
    months,
    online_sales,
    "o-",
    color="#1f77b4",
    label=label,
    markersize=8,
    linewidth=2,
)
axs[0].set_xlabel(xlabel_value, fontsize=12)
axs[0].set_ylabel(ylabel_value_1, fontsize=12)
axs[0].tick_params(axis="y", direction="in", labelsize=10)
axs[0].tick_params(axis="x", direction="in", labelsize=10)
axs[0].set_title(title_1)

# Plot the Store Sales data
(store_sales_line,) = axs[1].plot(
    months,
    store_sales,
    "s--",
    color="#2ca02c",
    label=label_1,
    markersize=8,
    linewidth=2,
)
axs[1].set_xlabel(xlabel_value, fontsize=12)
axs[1].set_ylabel(ylabel_value_2, fontsize=12)
axs[1].tick_params(axis="y", direction="in", labelsize=10)
axs[1].tick_params(axis="x", direction="in", labelsize=10)
axs[1].set_title(title_2)

# Add a legend to the plot
threshold = 200
axs[0].axhline(y=threshold, color="red", linestyle="-", linewidth=1.5, label=label_2)
axs[1].axhline(y=threshold, color="red", linestyle="-", linewidth=1.5, label=label_2)

# Highlight the data points above the threshold
for ax in axs:
    for month in highlight:
        ax.plot(
            month,
            online_sales[np.where(months == month)], "ro"
        )  # Highlight Online Sales
        ax.annotate(
            highlight_annotation,
            (month, online_sales[np.where(months == month)]),
            textcoords="offset points",
            xytext=(0, -20),
            ha="center",
        )

# Customize the plot with labels, title, and legend
axs[0].legend()
axs[1].legend()

# Add a grid to the plot
for ax in axs:
    ax.grid(True, linestyle="--", alpha=0.6)

# Add supertitle for the entire figure
plt.suptitle(supertitle, fontsize=16)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better spacing and display
plt.tight_layout(rect=[0, 0, 1, 0.96])
plt.savefig("line_278.pdf", bbox_inches="tight")
