
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Data for the plot
dosages = np.array([10, 20, 30, 40, 50, 60, 70])
medication_a_effectiveness = np.clip(
    np.log(dosages) + np.random.normal(0, 0.1, len(dosages)),
    1,
    5,
)
medication_b_effectiveness = np.clip(
    np.log(dosages + 5) + np.random.normal(0, 0.1, len(dosages)),
    1,
    5,
)
highlight = [20, 40, 60]

# Axes Limits and Labels
xlabel_value = "Dosage (mg)"
ylabel_value_1 = "Effectiveness (A)"
ylabel_value_2 = "Effectiveness (B)"

# Labels
label_1 = "Medication A"
label_2 = "Threshold"

label = "Medication B"

# Main title and subplot titles
main_title = "Effectiveness of Medications A and B Across Dosages"
subplot_title_1 = "Medication A Effectiveness"
subplot_title_2 = "Medication B Effectiveness"

# Annotation text
annotation_text = "Notable Point"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with a 1x2 grid
fig, axs = plt.subplots(1, 2, figsize=(14, 6))

fig.suptitle(main_title, fontsize=16)

# Plot Medication A data
(ax_a,) = axs[0].plot(
    dosages,
    medication_a_effectiveness,
    "o-",
    color="#1f77b4",
    label=label_1,
    markersize=8,
    linewidth=2,
)
axs[0].set_xlabel(xlabel_value, fontsize=12)
axs[0].set_ylabel(ylabel_value_1, fontsize=12)
axs[0].tick_params(axis="y", direction="in", labelsize=10)
axs[0].tick_params(axis="x", direction="in", labelsize=10)
axs[0].set_title(subplot_title_1, fontsize=14)

# Plot Medication B data
(ax_b,) = axs[1].plot(
    dosages,
    medication_b_effectiveness,
    "s--",
    color="green",
    label=label,
    markersize=8,
    linewidth=2,
)
axs[1].set_xlabel(xlabel_value, fontsize=12)
axs[1].set_ylabel(ylabel_value_2, fontsize=12)
axs[1].tick_params(axis="y", direction="in", labelsize=10)
axs[1].tick_params(axis="x", direction="in", labelsize=10)
axs[1].set_title(subplot_title_2, fontsize=14)

# Add a legend to the plot
threshold = 3.5
axs[0].axhline(y=threshold, color="red", linestyle="-", linewidth=1.5, label=label_2)
axs[1].axhline(y=threshold, color="red", linestyle="-", linewidth=1.5, label=label_2)

# Highlight the data points above the threshold
for ax in [axs[0], axs[1]]:
    for dose in highlight:
        ax.plot(
            dose, medication_a_effectiveness[np.where(dosages == dose)], "ro"
        ) if ax == axs[0] else ax.plot(
            dose, medication_b_effectiveness[np.where(dosages == dose)], "ro"
        )
        ax.annotate(
            annotation_text,
            (dose, medication_a_effectiveness[np.where(dosages == dose)])
            if ax == axs[0]
            else (dose, medication_b_effectiveness[np.where(dosages == dose)]),
            textcoords="offset points",
            xytext=(0, -20),
            ha="center",
        )

# Customize the plot with labels, title, and legend
axs[0].legend()
axs[1].legend()

# Add a grid to the plot
for ax in axs:
    ax.grid(True, linestyle="--", alpha=0.6)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better spacing and display
plt.tight_layout()
plt.savefig("line_280.pdf", bbox_inches="tight")
