
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(40)
# Data for the first plot (bar plot with white circles)
projects = [
    "Solar1",
    "Wind1",
    "Hydro1",
    "Biomass1",
    "Solar2",
    "Wind2",
    "Hydro2",
    "Biomass2",
    "Solar3",
    "Wind3",
    "Hydro3",
    "Biomass3",
    "Solar4",
    "Wind4",
    "Hydro4",
    "Biomass4",
]
n_projects = len(projects)
expected_output = np.random.uniform(100, 400, n_projects)
actual_output = expected_output - np.random.uniform(20, 100, n_projects)
sorted_indices = np.argsort((expected_output + actual_output) / 2)
projects = np.array(projects)[sorted_indices]
expected_output = expected_output[sorted_indices]
actual_output = actual_output[sorted_indices]

assignments = ["Solar", "Wind", "Hydro", "Biomass"]

# Data for the second plot (error bar plot with legend)
aggregate_efficiency = [np.random.randint(80, 100, size=5) for _ in range(len(assignments))]
variability = np.random.randint(5, 15, size=(len(assignments), 5))

# Data for the third plot (scatter plot with trend line and correct text)
costs = [
    np.linspace(1000, 5000, 5) - 1000 * i + np.random.randint(-200, 200, 5)
    for i in range(len(assignments))
]
efficiencies = [
    np.linspace(70, 95, 5) - 5 * i + np.random.randint(-10, 10, 5)
    for i in range(len(assignments))
]
scatterlabels = ["Expected", "Actual"]
xlabels = ["Energy Output (MWh)", "Aggregate Efficiency (%)", "Cost ($)"]
ylabels = ["Projects", "Efficiency (%)", "Efficiency (%)"]

ax2textlabel = "Project: Biomass"
title = "Renewable Energy Projects Overview"
text_j = 3500
text_i = 90

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Adjusting figure size to match the original image's dimensions
fig, axs = plt.subplots(1, 3, figsize=(12, 5))

# Colors and assignments for the first plot
colors = ["#f39c12", "#3498db", "#2ecc71", "#e74c3c"]

# First plot (bar plot with white circles)
axs[0].grid(axis="x", zorder=0)
# Plotting the lines first
for i in range(n_projects):
    color = colors[i // 4]
    axs[0].plot(
        [actual_output[i], expected_output[i]], [i, i], color=color, lw=6, zorder=1
    )
# Plotting the points
axs[0].scatter(
    expected_output,
    range(n_projects),
    color="black",
    label=scatterlabels[0],
    zorder=2,
    s=80,
)  # Black dots for 'Expected'
axs[0].scatter(
    actual_output,
    range(n_projects),
    color="white",
    edgecolors="black",
    label=scatterlabels[1],
    zorder=2,
    s=80,
)  # White dots for 'Actual'
axs[0].legend(loc="lower right", title="Output Type")
axs[0].set_xlabel(xlabels[0])
axs[0].set_ylabel(ylabels[0])
axs[0].set_yticks(range(n_projects))
axs[0].set_yticklabels(projects[::-1])

# Second plot (error bar plot with legend)
for i in range(len(assignments)):
    aggregate_eff = aggregate_efficiency[i]
    var = variability[i]
    axs[1].errorbar(
        np.linspace(1, 5, 5) * 10,
        aggregate_eff,
        yerr=var,
        fmt="o",
        capsize=3,
        color=colors[i],
    )
axs[1].set_xlabel(xlabels[1])
axs[1].set_ylabel(ylabels[1])
axs[1].grid()

# Third plot (scatter plot with trend line and correct text)
for i in range(len(assignments)):
    cost = costs[i]
    efficiency = efficiencies[i]
    error = np.random.randint(5, 10, size=5)
    axs[2].errorbar(
        cost,
        efficiency,
        yerr=error,
        xerr=error + 100,
        fmt="o",
        capsize=3,
        color=colors[i],
    )
# Plotting the trend line
axs[2].plot(
    np.linspace(1000, 5000, 5), np.linspace(50, 95, 5), color="black", lw=2, ls="-"
)
axs[2].set_xlabel(xlabels[2])
axs[2].set_ylabel(ylabels[2])
axs[2].text(
    text_j,
    text_i,
    ax2textlabel,
    fontsize=8,
    color="black",
    ha="right",
    va="bottom",
    bbox=dict(facecolor="white", edgecolor="black"),
)
axs[2].grid()

# add legend to the whole figure and set colormaps
legend_handles = [
    mlines.Line2D(
        [], [], color=color, marker="s", linestyle="None", markersize=10, label=label
    )
    for label, color in zip(assignments, colors)
]
fig.legend(
    handles=legend_handles,
    loc="upper center",
    title=title,
    ncol=len(assignments),
    bbox_to_anchor=(0.5, 1.15),
    facecolor="white",
)

# ===================
# Part 4: Saving Output
# ===================
# Show plot
plt.tight_layout()
plt.savefig("multidiff_50.pdf", bbox_inches="tight")
