
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)  # For reproducibility

# Data for the first plot (bar plot with white circles)
regions = ["North America", "Europe", "Asia", "Australia", "Africa", "South America"]
n_categories = len(regions)
temp_2000s = np.random.uniform(15, 25, n_categories)
temp_2020s = temp_2000s + np.random.uniform(-2, 2, n_categories)
sorted_indices = np.argsort(temp_2020s)
regions = np.array(regions)[sorted_indices]
temp_2000s = temp_2000s[sorted_indices]
temp_2020s = temp_2020s[sorted_indices]

models = ["Model A", "Model B", "Model C", "Model D"]

# Data for the second plot (error bar plot with legend)
for i in range(len(models)):
    temp_change = np.random.uniform(-1, 2, size=5)
    temp_model = np.random.uniform(18, 22, size=5)
    error = np.random.uniform(0.5, 1.5, size=5)

# Data for the third plot (scatter plot with trend line and correct text)
co2_levels = [
    np.linspace(300, 400, 5) + 20 * i + np.random.randint(-10, 10, 5)
    for i in range(len(models))
]
temp_changes = [
    np.linspace(0.5, 1.5, 5) + np.random.uniform(-0.5, 0.5, 5)
    for i in range(len(models))
]
scatterlabels = ["2000s", "2020s"]
xlabels = ["Temperature (°C)", "Model-based Temperature Change (°C)", "CO2 Levels (ppm)"]
ylabels = ["Region", "Temperature Change (°C)", "Temperature Change (°C)"]

ax2textlabel = "Climate Model: Model D"
title = "Temperature Changes by Region and Model"
text_j = 380
text_i = 1.5

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Adjusting figure size to match the original image's dimensions
fig, axs = plt.subplots(1, 3, figsize=(10, 4))

# Colors and assignments for the first plot
colors = ["#1f77b4", "#ff7f0e", "#2ca02c", "#d62728"]

# First plot (bar plot with white circles)
axs[0].grid(axis="x", zorder=0)
# Plotting the lines first
for i in range(n_categories):
    color = colors[i % 4]
    axs[0].plot(
        [temp_2000s[i], temp_2020s[i]], [i, i], color=color, lw=6, zorder=1
    )  # Line between the two points
# Plotting the points
axs[0].scatter(
    temp_2020s,
    range(n_categories),
    color="black",
    label=scatterlabels[0],
    zorder=2,
    s=80,
)  # Dots for 2020s
axs[0].scatter(
    temp_2000s,
    range(n_categories),
    color="white",
    edgecolors="black",
    label=scatterlabels[1],
    zorder=2,
    s=80,
)  # Dots for 2000s
axs[0].legend(loc="lower right", title="Decades")
axs[0].set_xlabel(xlabels[0])
axs[0].set_ylabel(ylabels[0])
axs[0].set_yticks(range(n_categories))
axs[0].set_yticklabels(regions[::-1])

# Second plot (error bar plot with legend)
for i in range(len(models)):
    axs[1].errorbar(
        temp_change,
        temp_model,
        yerr=error,
        xerr=error + 0.5,
        fmt="o",
        capsize=3,
        color=colors[i],
    )
axs[1].set_xlabel(xlabels[1])
axs[1].set_ylabel(ylabels[1])
axs[1].grid()

# Third plot (scatter plot with trend line and correct text)
for i in range(len(models)):
    co2_level = co2_levels[i]
    temp_change = temp_changes[i]
    error = np.random.uniform(0.2, 0.5, size=5)
    axs[2].errorbar(
        co2_level,
        temp_change,
        yerr=error,
        xerr=error + 2,
        fmt="o",
        capsize=3,
        color=colors[i],
    )
# Plotting the trend line
axs[2].plot(
    np.linspace(300, 400, 5), np.linspace(0.5, 2, 5), color="black", lw=2, ls="--"
)
axs[2].set_xlabel(xlabels[2])
axs[2].set_ylabel(ylabels[2])
# set box edge color
axs[2].text(
    text_j,
    text_i,
    ax2textlabel,
    fontsize=8,
    color="black",
    ha="right",
    va="bottom",
    bbox=dict(facecolor="white", edgecolor="black"),
)
axs[2].grid()

# add legend to the whole figure and set colormaps
legend_handles = [
    mlines.Line2D(
        [], [], color=color, marker="s", linestyle="None", markersize=10, label=label
    )
    for label, color in zip(models, colors)
]
fig.legend(
    handles=legend_handles,
    loc="upper center",
    title=title,
    ncol=len(models),
    bbox_to_anchor=(0.5, 1.15),
    facecolor="white",
)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("multidiff_52.pdf", bbox_inches="tight")
