
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.cm import ScalarMappable
from matplotlib.colors import Normalize

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)
viewer_engagement_platform_a = np.random.rand(12, 12) * 20 + 50  # Simulated viewer engagement scores, platform A
viewer_engagement_platform_b = np.random.rand(12, 12) * 20 + 45  # Simulated viewer engagement scores, platform B
bar_data_platform_a = viewer_engagement_platform_a.mean(axis=0)
bar_data_platform_b = viewer_engagement_platform_b.mean(axis=0)

# Attributes for y-axis
media_attributes = [
    "Action",
    "Comedy",
    "Drama",
    "Sci-Fi",
    "Romance",
    "Documentary",
    "Horror",
    "Animation",
    "Thriller",
    "Fantasy",
    "Mystery",
    "Musical",
]
heatmap_xlabel = "Media Attribute"
bar_xlabel = "Average Engagement Score"
bar_xticks = [40, 50, 60, 70]
bar_xlim = [40, 70]

# Normalize the bar data to match heatmap's vmin and vmax
norm = Normalize(vmin=40, vmax=70)

# Text labels
heatmap_top_label = "Platform A"
heatmap_bottom_label = "Platform B"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
cmap = plt.get_cmap("viridis")
# Create figure and axes
fig = plt.figure(figsize=(10, 8))

# Heatmap top
ax_heatmap_top = fig.add_subplot(2, 2, 1)
cax_top = ax_heatmap_top.imshow(
    viewer_engagement_platform_a, cmap=cmap, aspect="auto", vmin=40, vmax=70
)
ax_heatmap_top.set_yticks(np.arange(len(media_attributes)))
ax_heatmap_top.set_yticklabels(media_attributes)
ax_heatmap_top.set_xticks([])
ax_heatmap_top.set_title(heatmap_top_label)

# Create an axes on the top side of ax_heatmap_top for the colorbar.
ax_colorbar = fig.add_axes(
    [
        ax_heatmap_top.get_position().x0,
        ax_heatmap_top.get_position().y1 + 0.15,
        ax_heatmap_top.get_position().width,
        0.02,
    ]
)
ax_colorbar.xaxis.set_ticks_position("top")
ax_colorbar.xaxis.set_label_position("top")

# Adding a colorbar at the very top of the heatmap
plt.colorbar(cax_top, cax=ax_colorbar, orientation="horizontal")

# Heatmap bottom
ax_heatmap_bottom = fig.add_subplot(2, 2, 3)
cax_bottom = ax_heatmap_bottom.imshow(
    viewer_engagement_platform_b, cmap=cmap, aspect="auto", vmin=40, vmax=70
)
ax_heatmap_bottom.set_yticks(np.arange(len(media_attributes)))
ax_heatmap_bottom.set_yticklabels(media_attributes)
ax_heatmap_bottom.set_xticks(np.arange(len(media_attributes)))
ax_heatmap_bottom.set_xticklabels(media_attributes, rotation=45, ha="right")
ax_heatmap_bottom.set_xlabel(heatmap_xlabel)
ax_heatmap_bottom.set_title(heatmap_bottom_label)

# Bar plot top
ax_bar_top = fig.add_subplot(2, 2, 2)
colors_top = [cmap(norm(value)) for value in bar_data_platform_a]
ax_bar_top.barh(np.arange(len(media_attributes)), bar_data_platform_a, color=colors_top)
ax_bar_top.set_yticks([])
ax_bar_top.set_xticks([])
ax_bar_top.set_xlim(bar_xlim)
ax_bar_top.set_ylim(ax_heatmap_top.get_ylim())

# Bar plot bottom
ax_bar_bottom = fig.add_subplot(2, 2, 4)
colors_bottom = [cmap(norm(value)) for value in bar_data_platform_b]
ax_bar_bottom.barh(np.arange(len(media_attributes)), bar_data_platform_b, color=colors_bottom)
ax_bar_bottom.set_yticks([])
ax_bar_bottom.set_xticks(bar_xticks)
ax_bar_bottom.set_xlim(bar_xlim)
ax_bar_bottom.set_xlabel(bar_xlabel)
ax_bar_bottom.set_ylim(ax_heatmap_bottom.get_ylim())

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save plot
plt.tight_layout()  # leave some space at the top for the colorbar
plt.savefig("multidiff_62.pdf", bbox_inches="tight")
