
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# ErrorBar Plot Data
# Historical Periods
periods = ["Medieval", "Renaissance", "Industrial", "Modern"]
# Hypothetical event likelihood (percentage)
event_likelihood = [35, 45, 60, 80]
# Standard errors for the above likelihood
errors = [5, 6, 7, 8]

# ErrorPoint Plot Data
# Types of historical events
event_types = ["Battles", "Inventions", "Treaties", "Explorations"]
# Randomly generated popularity rates and standard deviations
popularity_rates = np.random.uniform(0.1, 1.0, len(event_types))
std_devs = np.random.uniform(0.01, 0.05, len(event_types))
dataset_mean = np.mean(popularity_rates)
titles = ["Event Likelihood by Historical Period", "Event Popularity by Type"]
ylabels = ["Likelihood (%)", "Popularity Rate"]
ylim = [0, 1.2]
xlabel_events = "Event Types"
legendlabels = ["Popularity Rates", "Mean Popularity Rate"]
subtitle = "Historical Data Visualizations"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axes for the subplots
fig, axes = plt.subplots(2, 1, figsize=(10, 12))
fig.suptitle(subtitle, fontsize=16)

# ErrorBar Plot
axes[0].bar(
    periods, event_likelihood, yerr=errors, color="#D2B48C", capsize=5, ecolor="brown"
)
axes[0].set_title(titles[0])
axes[0].set_ylabel(ylabels[0])
axes[0].grid(True, linestyle='--', linewidth=0.5)

# ErrorPoint Plot
axes[1].errorbar(
    event_types,
    popularity_rates,
    yerr=std_devs,
    fmt="D",
    color="#8B4513",
    ecolor="#8B4513",
    capsize=5,
    ms=8,
    linestyle='None'
)
axes[1].axhline(y=dataset_mean, color="grey", linestyle="--")
axes[1].set_title(titles[1])
axes[1].set_ylabel(ylabels[1])
axes[1].set_xlabel(xlabel_events)
axes[1].set_ylim(ylim)
axes[1].legend(legendlabels, loc='upper left')
axes[1].grid(True, linestyle='--', linewidth=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to avoid overlap and save the figure
plt.tight_layout(rect=[0, 0.03, 1, 0.97])
plt.savefig("multidiff_80.pdf", bbox_inches="tight")
