
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Seed for reproducibility
np.random.seed(0)

# Generating synthetic data for the violin plot (Temperature Trends)
forest_temps = np.random.normal(22, 5, 150)  # Forest ecosystem temperatures
desert_temps = np.random.normal(30, 7, 150)  # Desert ecosystem temperatures

# Generating synthetic data for the density plot (Precipitation Levels)
coastal_precip = np.random.normal(120, 30, 1000)  # Coastal region precipitation
inland_precip = np.random.normal(90, 25, 1000)  # Inland region precipitation
mountain_precip = np.random.normal(150, 35, 1000)  # Mountainous region precipitation

# Defining x-axis range for density plots
xs = np.linspace(0, 250, 200)

# Labels and titles for the plots
labels = ["Coastal", "Inland", "Mountainous"]
titles = ["Average Temperatures in Different Ecosystems", "Precipitation Distribution in Various Regions"]
ylabels = ["Temperature (°C)", "Density"]
ax1xticks = [1, 2]
ax1xtickslabels = ["Forest", "Desert"]
ax2xlabel = "Precipitation (mm)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating the figure and axes
fig, axes = plt.subplots(nrows=1, ncols=2, figsize=(12, 6))

# Violin plot
violin_data = [forest_temps, desert_temps]
axes[0].violinplot(violin_data, showmeans=False, showmedians=True, 
                   showextrema=False, bw_method=0.5)
axes[0].set_title(titles[0], fontsize=14, weight='bold')
axes[0].set_ylabel(ylabels[0], fontsize=12)
axes[0].set_xticks(ax1xticks)
axes[0].set_xticklabels(ax1xtickslabels, fontsize=10)
axes[0].grid(True)

# Density plot
colors = ["#1f77b4", "#ff7f0e", "#2ca02c"]  # Custom colors for each region
line_styles = ['-', '--', '-.']
for data, color, label, linestyle in zip([coastal_precip, inland_precip, mountain_precip], colors, labels, line_styles):
    density = gaussian_kde(data)
    axes[1].plot(xs, density(xs), color=color, linestyle=linestyle, linewidth=2, label=label)
    axes[1].fill_between(xs, density(xs), color=color, alpha=0.2)
axes[1].set_title(titles[1], fontsize=14, weight='bold')
axes[1].set_xlabel(ax2xlabel, fontsize=12)
axes[1].set_ylabel(ylabels[1], fontsize=12)
axes[1].legend(fontsize=10)
axes[1].grid(True)

# ===================
# Part 4: Saving Output
# ===================
# Adjusting layout for better visual appearance
plt.tight_layout()
plt.savefig("multidiff_85.pdf", bbox_inches="tight")
