
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.gridspec import GridSpec

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# ========== Bar with Error Data (Average Delay Times) ==========
# Modes of transportation
transport_modes = ["Bus", "Train", "Car", "Bike"]
# Average delay times (in minutes)
delay_times = [10, 5, 15, 2]
# Error margins
delay_errors = [2, 1, 3, 0.5]

# ========== Scatter with Error Data (Fuel Efficiency) ==========
# Vehicle Types
vehicle_types = ["Sedan", "SUV", "Truck", "Motorcycle"]
# Fuel efficiency scores (miles per gallon)
fuel_efficiency = np.random.uniform(15, 30, len(vehicle_types))
# Standard deviations
fuel_eff_std_devs = np.random.uniform(1, 3, len(vehicle_types))

# ========== Line with Error Data (Pollution Levels) ==========
# Vehicle Models
vehicle_models = ["Model X", "Model Y", "Model Z", "Model W"]
# Pollution levels over time (in grams per mile)
pollution_level_2020 = np.random.uniform(200, 300, len(vehicle_models))
pollution_level_2021 = np.random.uniform(180, 280, len(vehicle_models))
# Error
pollution_error = np.random.uniform(10, 20, len(vehicle_models))

# Titles and Labels
titles = [
    "Average Delay Times by Transportation Mode",
    "Fuel Efficiency Scores by Vehicle Type",
    "Vehicle Pollution Levels Over Time",
]
ylabels = ["Delay Time (minutes)", "Fuel Efficiency (mpg)", "Pollution Level (g/mile)"]
ax2ylim = [10, 35]
ax3labels = ["2020", "2021"]

legend_title = "Year"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and grid layout
fig = plt.figure(figsize=(10, 10))
gs = GridSpec(2, 2, figure=fig)

# Bar with Error Plot
ax1 = fig.add_subplot(gs[0, :])
ax1.bar(transport_modes, delay_times, yerr=delay_errors, color="steelblue", capsize=5, ecolor="grey")
ax1.set_title(titles[0])
ax1.set_ylabel(ylabels[0])
ax1.grid(True)

# Scatter with Error Plot
ax2 = fig.add_subplot(gs[1, 0])
ax2.errorbar(
    vehicle_types,
    fuel_efficiency,
    yerr=fuel_eff_std_devs,
    fmt="o",
    color="orange",
    ecolor="grey",
    capsize=5,
    ms=10,
)
ax2.set_title(titles[1])
ax2.set_ylabel(ylabels[1])
ax2.set_ylim(ax2ylim)
ax2.grid(True)

# Line with Error Plot
ax3 = fig.add_subplot(gs[1, 1])
ax3.errorbar(
    vehicle_models,
    pollution_level_2020,
    yerr=pollution_error,
    fmt="^-",
    color="green",
    ecolor="grey",
    capsize=5,
    label=ax3labels[0],
)
ax3.errorbar(
    vehicle_models,
    pollution_level_2021,
    yerr=pollution_error,
    fmt="s-",
    color="blue",
    ecolor="grey",
    capsize=5,
    label=ax3labels[1],
)
ax3.set_title(titles[2])
ax3.set_ylabel(ylabels[2])
ax3.legend(title=legend_title)
ax3.grid(True)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout()
plt.savefig("multidiff_90.pdf", bbox_inches="tight")
