
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde
from matplotlib.gridspec import GridSpec

# ===================
# Part 2: Data Preparation
# ===================
# Seed for reproducibility
import numpy as np

np.random.seed(0)

# Generate synthetic agriculture data
# Crop yields (in tonnes per hectare) for different crops
crop_yields_wheat = np.random.poisson(30, 500)  # Yield of wheat
crop_yields_corn = np.random.poisson(45, 500)  # Yield of corn

# Soil moisture levels (percentage)
soil_moisture_zone1 = np.random.normal(20, 5, 1000)  # Zone 1 soil moisture
soil_moisture_zone2 = np.random.normal(30, 5, 1000)  # Zone 2 soil moisture

# Temperature data across a growing season (days)
days = np.linspace(0, 120, 1000)
temperature_variation = 25 + 10 * np.sin(np.linspace(0, 2 * np.pi, 1000))

ax1labels = ["Wheat Yield", "Corn Yield"]
titles = [
    "Crop Yields",
    "Soil Moisture Levels",
    "Temperature Variation During Growing Season",
]
xlabels = ["Yield (tonnes/ha)", "Days"]
ylabels = ["Frequency", "Moisture (%)", "Temperature (°C)"]
ax2xtickslabels = ["Zone 1", "Zone 2"]
ax2xticks = [1, 2]
bins = np.linspace(10, 60, 20)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the figure using GridSpec
fig = plt.figure(figsize=(10, 10))
gs = GridSpec(2, 2, figure=fig)

# Histogram plot across top (1,1 and 1,2)
ax1 = fig.add_subplot(gs[0, :])
ax1.hist(
    crop_yields_wheat,
    bins=bins,
    alpha=0.7,
    label=ax1labels[0],
    color="#7fbc41",
    edgecolor="black",
)
ax1.hist(
    crop_yields_corn,
    bins=bins,
    alpha=0.7,
    label=ax1labels[1],
    color="#b3de69",
    edgecolor="black",
)
ax1.set_title(titles[0])
ax1.set_xlabel(xlabels[0])
ax1.set_ylabel(ylabels[0])
ax1.legend()

# Violin plot on bottom left (2,1)
ax2 = fig.add_subplot(gs[1, 0])
ax2.violinplot([soil_moisture_zone1, soil_moisture_zone2], showmeans=False, showmedians=True)
ax2.set_title(titles[1])
ax2.set_ylabel(ylabels[1])
ax2.set_xticks(ax2xticks)
ax2.set_xticklabels(ax2xtickslabels)
ax2.grid(True)

# Fill between plot on bottom right (2,2)
ax3 = fig.add_subplot(gs[1, 1])
ax3.fill_between(days, temperature_variation, color="brown", alpha=0.2)
ax3.set_title(titles[2])
ax3.set_xlabel(xlabels[1])
ax3.set_ylabel(ylabels[2])

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for a better visual appearance
plt.tight_layout()
plt.savefig("multidiff_92.pdf", bbox_inches="tight")
