# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines
import numpy as np

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot
countries_1 = ["USA", "UK", "Canada"]
gov_start_1 = [6, 5.5, 6.5]
gov_bias_1 = [-3.2, -2.7, -3.5]
private_start_1 = [7, 6.5, 7.5]
private_bias_1 = [+1.5, +1.8, +1.2]
ax1_labels = ["Government Trust Bias", "Private Sector Trust Bias"]

countries_2 = ["Germany", "France", "Italy"]
gov_start_2 = [5.8, 6.1, 5.9]
gov_bias_2 = [-2.8, -3.3, -2.9]
private_start_2 = [6.8, 7.0, 6.7]
private_bias_2 = [+1.1, +1.9, +1.0]
ax2_labels = ["Government Trust Bias", "Private Sector Trust Bias"]

# Chart labels
chart_title = "Trust Levels in Different Institutions (Bias Index)"
x_label = "Trust Score"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 5))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# Custom colors
gov_color = "brown"
private_color = "green"

# First subplot (countries_1)
for i, country in enumerate(countries_1):
    # Government bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(gov_start_1[i], i + offset * 3 / 2),
        xytext=(gov_start_1[i] + gov_bias_1[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color=gov_color),
    )
    ax1.scatter(
        [gov_start_1[i], gov_start_1[i] + gov_bias_1[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{gov_bias_1[i]:.2f}",
        (gov_start_1[i] + gov_bias_1[i], i + offset * 1.75),
        color=gov_color,
        ha="right",
        va="center",
    )

    # Private sector bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(private_start_1[i], i + offset / 2),
        xytext=(private_start_1[i] + private_bias_1[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<-", color=private_color),
    )
    ax1.scatter(
        [private_start_1[i], private_start_1[i] + private_bias_1[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{private_bias_1[i]:.2f}",
        (private_start_1[i] + private_bias_1[i], i + offset * 0.75),
        color=private_color,
        ha="left",
        va="center",
    )

# Second subplot (countries_2)
for i, country in enumerate(countries_2):
    ax2.annotate(
        "",
        xy=(gov_start_2[i], i + offset * 3 / 2),
        xytext=(gov_start_2[i] + gov_bias_2[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color=gov_color),
    )
    ax2.scatter(
        [gov_start_2[i], gov_start_2[i] + gov_bias_2[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{gov_bias_2[i]:.2f}",
        (gov_start_2[i] + gov_bias_2[i], i + offset * 1.75),
        color=gov_color,
        ha="right",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(private_start_2[i], i + offset / 2),
        xytext=(private_start_2[i] + private_bias_2[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<-", color=private_color),
    )
    ax2.scatter(
        [private_start_2[i], private_start_2[i] + private_bias_2[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{private_bias_2[i]:.2f}",
        (private_start_2[i] + private_bias_2[i], i + offset * 0.75),
        color=private_color,
        ha="left",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(countries_1))
ax2.set_ylim(0, len(countries_2))

# Set x-axis limits uniformly
ax1.set_xlim(0, 10)
ax2.set_xlim(0, 10)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(countries_1))])
ax1.set_yticklabels(countries_1)
ax2.set_yticks([i + offset for i in range(len(countries_2))])
ax2.set_yticklabels(countries_2)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(countries_1))], minor=True)
ax2.set_yticks([i for i in range(len(countries_2))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")

# Add x-axis grid lines and set gap to 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
gov_arrow = mlines.Line2D(
    [],
    [],
    color=gov_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax1_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
private_arrow = mlines.Line2D(
    [],
    [],
    color=private_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax1_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[gov_arrow, private_arrow], bbox_to_anchor=(0.45, 0), ncol=2)

gov_arrow = mlines.Line2D(
    [],
    [],
    color=gov_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax2_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
private_arrow = mlines.Line2D(
    [],
    [],
    color=private_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax2_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[gov_arrow, private_arrow], bbox_to_anchor=(0.85, 0), ncol=2)
# Super title and x label
plt.suptitle(chart_title)
fig.text(0.5, 0.01, x_label, ha="center")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout(rect=[0, 0.03, 1, 0.95])
plt.savefig("quiver_21.pdf", bbox_inches="tight")
