# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot
# Set 1: Solar Energy
countries_1 = ["USA", "Germany", "China"]
solar_start = [8, 7.5, 9]
solar_growth = [2.5, 3, 2]

# Set 2: Wind Energy
countries_2 = ["India", "Spain", "Brazil"]
wind_start = [6, 5.5, 6.5]
wind_growth = [1.8, 2.2, 1.6]

# Axes Limits and Labels
xlim_values = [0, 12]
xlabel = "Time (Years)"
ylabel = "Energy Production (GW)"
title = "Renewable Energy Production Growth"
label_solar = "Solar Energy Growth"
label_wind = "Wind Energy Growth"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with a single subplot
fig, ax1 = plt.subplots(figsize=(7, 5))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (countries_1)
for i, country in enumerate(countries_1):
    # Solar Energy growth line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(solar_start[i], i + offset * 3 / 2),
        xytext=(solar_start[i] + solar_growth[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color="green"),
    )
    ax1.scatter(
        [solar_start[i], solar_start[i] + solar_growth[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{solar_growth[i]:.1f}GW",
        (solar_start[i] + solar_growth[i], i + offset * 1.75),
        color="green",
        ha="right",
        va="center",
    )

# Second subplot (countries_2)
for i, country in enumerate(countries_2):
    # Wind Energy growth line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(wind_start[i], len(countries_1) + i + offset * 3 / 2),
        xytext=(wind_start[i] + wind_growth[i], len(countries_1) + i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color="blue"),
    )
    ax1.scatter(
        [wind_start[i], wind_start[i] + wind_growth[i]],
        [len(countries_1) + i + offset * 3 / 2, len(countries_1) + i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{wind_growth[i]:.1f}GW",
        (wind_start[i] + wind_growth[i], len(countries_1) + i + offset * 1.75),
        color="blue",
        ha="right",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(countries_1) + len(countries_2))

# Set x-axis limits uniformly
ax1.set_xlim(xlim_values)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(countries_1) + len(countries_2))])
ax1.set_yticklabels(countries_1 + countries_2)

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(countries_1) + len(countries_2))], minor=True)

ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")

# Add x-axis grid lines and set gap to 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))

ax1.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
green_arrow = mlines.Line2D(
    [],
    [],
    color="green",
    marker=">",
    linestyle="-",
    markersize=8,
    label=label_solar,
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)

blue_arrow = mlines.Line2D(
    [],
    [],
    color="blue",
    marker=">",
    linestyle="-",
    markersize=8,
    label=label_wind,
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)

# Set title and labels
ax1.set_xlabel(xlabel)
ax1.set_ylabel(ylabel)
ax1.set_title(title)

fig.legend(handles=[green_arrow, blue_arrow], bbox_to_anchor=(0.5, -0.1), ncol=1)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_24.pdf", bbox_inches="tight")
