# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================

# New Business Data for the plot
departments = ["Sales", "Marketing", "Finance"]
out_start_business = [8.0, 7.5, 9.0]
out_group_bias_business = [-3.5, -2.75, -4.0]
in_start_business = [7.0, 6.5, 8.5]
in_group_bias_business = [+2.5, +2.0, +3.0]

# Axes Limits and Labels
xlim_business = [0, 10]
label_business = "Out-group bias\n(Department)"

# Chart title and axis labels
chart_title = "Departmental Performance and Bias"
xlabel = "Performance Metrics"
ylabel = "Departments"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with a single subplot
fig, ax1 = plt.subplots(figsize=(5, 4))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (departments)
for i, dept in enumerate(departments):
    # Out-group bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(out_start_business[i], i + offset * 3 / 2),
        xytext=(out_start_business[i] + out_group_bias_business[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color="blue"),
    )
    ax1.scatter(
        [out_start_business[i], out_start_business[i] + out_group_bias_business[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{out_group_bias_business[i]:.2f}",
        (out_start_business[i] + out_group_bias_business[i], i + offset * 1.75),
        color="blue",
        ha="right",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(departments))

# Set x-axis limits uniformly
ax1.set_xlim(xlim_business)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(departments))])
ax1.set_yticklabels(departments)

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(departments))], minor=True)

ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")

# Add x-axis grid lines and set the gap to be 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))

ax1.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
blue_arrow = mlines.Line2D(
    [],
    [],
    color="blue",
    marker=">",
    linestyle="-",
    markersize=8,
    label=label_business,
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)

fig.legend(handles=[blue_arrow], bbox_to_anchor=(0.45, 0), ncol=2)

# Add title and labels
ax1.set_title(chart_title)
ax1.set_xlabel(xlabel)
ax1.set_ylabel(ylabel)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_25.pdf", bbox_inches="tight")
