# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot
subjects_1 = ["Fiction", "Non-Fiction", "Poetry"]
out_start_1 = [6.5, 7.9, 8.0]
out_group_bias_1 = [-1.5, -2.0, -1.8]
in_start_1 = [6.8, 7.5, 7.2]
in_group_bias_1 = [+1.2, +1.5, +1.8]
ax1_labels = ["Out-group bias\n(Non-Fiction)", "In-group bias\n(Non-Fiction)"]

subjects_2 = ["Fantasy", "Science Fiction", "Mystery"]
out_start_2 = [8.0, 8.8, 6.9]
out_group_bias_2 = [-2.0, -1.5, -2.2]
in_start_2 = [7.2, 8.4, 6.5]
in_group_bias_2 = [+1.4, +1.8, +1.2]
ax2_labels = ["Out-group bias\n(Science Fiction)", "In-group bias\n(Science Fiction)"]

xlabel = "Rating (out of 10)"
ylabel = "Genres"
suptitle = "Bias Analysis in Literature Genres"
title_ax1 = "General Literature"
title_ax2 = "Genre-Specific Literature"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(14, 6))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# Custom colors
out_color = "saddlebrown"  # out-group bias color
in_color = "olivedrab"  # in-group bias color
dot_color = "peru"  # dot color

# First subplot (subjects_1)
for i, subject in enumerate(subjects_1):
    # Out-group bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(out_start_1[i], i + offset * 3 / 2),
        xytext=(out_start_1[i] + out_group_bias_1[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color=out_color),
    )
    ax1.scatter(
        [out_start_1[i], out_start_1[i] + out_group_bias_1[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color=dot_color,
        s=20,
    )
    ax1.annotate(
        f"{out_group_bias_1[i]:.2f}",
        (out_start_1[i] + out_group_bias_1[i], i + offset * 1.75),
        color=out_color,
        ha="right",
        va="center",
    )

    # In-group bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(in_start_1[i], i + offset / 2),
        xytext=(in_start_1[i] + in_group_bias_1[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<-", color=in_color),
    )
    ax1.scatter(
        [in_start_1[i], in_start_1[i] + in_group_bias_1[i]],
        [i + offset / 2, i + offset / 2],
        color=dot_color,
        s=20,
    )
    ax1.annotate(
        f"{in_group_bias_1[i]:.2f}",
        (in_start_1[i] + in_group_bias_1[i], i + offset * 0.75),
        color=in_color,
        ha="left",
        va="center",
    )

# Second subplot (subjects_2)
for i, subject in enumerate(subjects_2):
    ax2.annotate(
        "",
        xy=(out_start_2[i], i + offset * 3 / 2),
        xytext=(out_start_2[i] + out_group_bias_2[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color=out_color),
    )
    ax2.scatter(
        [out_start_2[i], out_start_2[i] + out_group_bias_2[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color=dot_color,
        s=20,
    )
    ax2.annotate(
        f"{out_group_bias_2[i]:.2f}",
        (out_start_2[i] + out_group_bias_2[i], i + offset * 1.75),
        color=out_color,
        ha="right",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(in_start_2[i], i + offset / 2),
        xytext=(in_start_2[i] + in_group_bias_2[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<-", color=in_color),
    )
    ax2.scatter(
        [in_start_2[i], in_start_2[i] + in_group_bias_2[i]],
        [i + offset / 2, i + offset / 2],
        color=dot_color,
        s=20,
    )
    ax2.annotate(
        f"{in_group_bias_2[i]:.2f}",
        (in_start_2[i] + in_group_bias_2[i], i + offset * 0.75),
        color=in_color,
        ha="left",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(subjects_1))
ax2.set_ylim(0, len(subjects_2))

# Set x-axis limits uniformly
ax1.set_xlim(4, 10)
ax2.set_xlim(4, 10)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(subjects_1))])
ax1.set_yticklabels(subjects_1)
ax2.set_yticks([i + offset for i in range(len(subjects_2))])
ax2.set_yticklabels(subjects_2)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(subjects_1))], minor=True)
ax2.set_yticks([i for i in range(len(subjects_2))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")

# Add x-axis grid lines and set gap is 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Add titles and labels
ax1.set_title(title_ax1, fontsize=12)
ax2.set_title(title_ax2, fontsize=12)
fig.suptitle(suptitle, fontsize=16)
fig.supxlabel(xlabel, fontsize=14)
fig.supylabel(ylabel, fontsize=14)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
out_legend = mlines.Line2D(
    [],
    [],
    color=out_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax1_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
in_legend = mlines.Line2D(
    [],
    [],
    color=in_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax1_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[out_legend, in_legend], bbox_to_anchor=(0.55, -0.05), ncol=2)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_38.pdf", bbox_inches="tight")
