# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Data
categories = [
    "Stoicism",
    "Existentialism",
    "Nihilism",
    "Utilitarianism",
    "Deontology",
    "Virtue Ethics",
]
values1 = [55.5, 60.3, 45.2, 70.1, 58.7, 65.4]
values2 = [45.4, 40.2, 40.5, 45.3, 45.8, 40.9]
labels = ["Philosopher A", "Philosopher B"]

# Text labels
title = "Philosophical Schools Performance"
xlabel = "Philosophical Dimensions"
legendlabels = [labels[0], labels[1]]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Number of variables
num_vars = len(categories)

# Compute angle for each category
angles = np.linspace(0, 2 * np.pi, num_vars, endpoint=False).tolist()

# The plot is circular, so we need to "complete the loop" and append the start to the end.
values1 += values1[:1]
values2 += values2[:1]
angles += angles[:1]

# Color scheme
color1 = "#734f96"
color2 = "#88c999"

# Plot
fig, ax = plt.subplots(figsize=(8, 7), subplot_kw=dict(polar=True))
ax.fill(angles, values1, color=color1, alpha=0.25)
ax.fill(angles, values2, color=color2, alpha=0.25)
ax.plot(
    angles,
    values1,
    color=color1,
    linewidth=2,
    linestyle="-",
    marker="o",
    label=labels[0],
)
ax.plot(
    angles,
    values2,
    color=color2,
    linewidth=2,
    linestyle="--",
    marker="s",
    label=labels[1],
)

# Labels and annotations for each point
for angle, value1, value2 in zip(angles[:-1], values1[:-1], values2[:-1]):
    ax.annotate(
        f"{value1}",
        xy=(angle, value1),
        xytext=(5, 5),
        textcoords="offset points",
        fontsize=9,
        bbox=dict(boxstyle="round,pad=0.3", edgecolor=color1, facecolor="w"),
    )
    ax.annotate(
        f"{value2}",
        xy=(angle, value2),
        xytext=(5, -10),
        textcoords="offset points",
        fontsize=9,
        bbox=dict(boxstyle="round,pad=0.3", edgecolor=color2, facecolor="w"),
    )

# Labels for each point
ax.set_xticks(angles[:-1])
ax.set_xticklabels(categories, fontsize=10, color="#333333")

# Remove ylabels
ax.set_yticklabels([])

# Title
plt.title(title, fontsize=14, color="#333333")

# Legend
ax.legend(
    loc="lower center",
    bbox_to_anchor=(0.5, -0.15),
    ncol=2,
    frameon=True,
    facecolor="#f2f2f2",
    edgecolor="#f2f2f2",
    fontsize=10,
)

ax.xaxis.set_tick_params(pad=25)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit and save the plot
plt.tight_layout()
plt.savefig("radar_45.pdf", bbox_inches="tight")
