# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines
import numpy as np

# ===================
# Part 2: Data Preparation
# ===================
# New dataset for Financial Analysis focusing on Investment Methods
methods = ["Stocks", "Bonds", "Real Estate", "Cryptocurrency", "Mutual Funds", "ETFs"]

# Data for the subplots representing two different timelines.
diversity_data = [[0.8, 2.1, 1.9, 1.5, 2.3, 1.6], [1.5, 3.2, 5.0, 4.5, 3.8, 2.6]]
accuracy_data = [[55, 60, 45, 75, 50, 80], [65, 70, 85, 80, 75, 90]]
scatter_sizes = [80, 120, 100, 160, 200, 180]

# Legend labels for the subplots.
ax1_legend_names = ["1 Year", "5 Years", "10 Years", "20 Years"]
ax2_legend_names = ["Short-term", "Medium-term", "Long-term", "Very Long-term"]
ax1_legend_title = "Investment Horizon"
ax2_legend_title = "Investment Period"
xlabel = "Portfolio Diversification"
ylabel = "Return Accuracy (%)"
suptitle = "Performance of Investment Methods Over Time"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create 2x1 grid of subplots with a specified figure size.
fig, axs = plt.subplots(2, 1, figsize=(8, 8))

# Use new colors for the points to distinguish them better.
colors = ["#1f77b4", "#ff7f0e", "#2ca02c", "#d62728", "#9467bd", "#8c564b"]

# Populate the subplots with scatter points and add text labels.
for idx, ax in enumerate(axs):
    for method, div, acc, size, color in zip(
        methods, diversity_data[idx], accuracy_data[idx], scatter_sizes, colors
    ):
        ax.scatter(
            div, acc, s=size, color=color, alpha=0.6, edgecolors="w", linewidth=1
        )  # Plot the scatter points.
        ax.text(
            div, acc + 2, method, fontsize=9, ha="center", color="black", weight="bold"
        )  # Add text labels above scatter points.

    ax.set_xlabel(xlabel, fontsize=12)  # X-axis label.
    ax.set_ylabel(ylabel, fontsize=12)  # Y-axis label.
    ax.grid(True, linestyle="--", alpha=0.6)  # Add grid lines for better readability.
    ax.tick_params(
        axis="both", which="major", labelsize=10
    )  # Customize tick parameters.

# Adjust the x and y limits and ticks for the subplots.
axs[0].set_xlim(0.5, 2.5)
axs[0].set_ylim(40, 90)
axs[0].set_xticks(np.arange(0.5, 2.6, 0.5))
axs[0].set_yticks(np.arange(40, 91, 10))
axs[1].set_xlim(1.0, 6.0)
axs[1].set_ylim(60, 100)
axs[1].set_xticks(np.arange(1.0, 6.1, 1.0))
axs[1].set_yticks(np.arange(60, 101, 10))

size_legend_handles = [80, 120, 160, 200]  # Sizes for the legend handles.

# Create custom legend handles for the first subplot.
ax1_legend_handles = [
    mlines.Line2D(
        [],
        [],
        color=color,
        marker="o",
        linestyle="None",
        markersize=(size**0.5) * 0.8,
        label=name,
    )
    for size, name, color in zip(
        size_legend_handles, ax1_legend_names, colors[: len(ax1_legend_names)]
    )
]

# Create custom legend handles for the second subplot.
ax2_legend_handles = [
    mlines.Line2D(
        [],
        [],
        color=color,
        marker="o",
        linestyle="None",
        markersize=(size**0.5) * 0.8,
        label=name,
    )
    for size, name, color in zip(
        size_legend_handles, ax2_legend_names, colors[: len(ax2_legend_names)]
    )
]

# Add legends to the subplots.
axs[0].legend(
    handles=ax1_legend_handles,
    loc="upper left",
    title=ax1_legend_title,
    labelspacing=1.5,
    edgecolor="gray",
)
axs[1].legend(
    handles=ax2_legend_handles,
    loc="upper left",
    title=ax2_legend_title,
    labelspacing=1.5,
    edgecolor="gray",
)

# Add a common title for the entire figure
fig.suptitle(suptitle, fontsize=14, fontweight="bold")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout(rect=[0, 0, 1, 0.96])
plt.savefig("scatter_33.pdf", bbox_inches="tight")
